import type { CSSProperties } from 'react';
import type { CSSObject, FlattenSimpleInterpolation } from 'styled-components';
import styled, { css, keyframes } from 'styled-components';

const HIDDEN_OPACITY = 0.001;
const HIDE_TIMING_FUNC = 'ease-in-out';
const DEFAULT_HIDE_TRANSITION_DURATION = 0.25;

export const DEFAULT_HIDE_TRANSITION = `${DEFAULT_HIDE_TRANSITION_DURATION}s ${HIDE_TIMING_FUNC}`;

type AccessiblyHideProps = {
  /**
   * Extra CSS properties to coordinate transition timing with.
   */
  $extraTransitionProperties?: Array<keyof CSSProperties>;
  $hide: boolean;
  $transitionDurationSeconds?: number;
};

// istanbul ignore next: trivial
/**
 * Allows an element to Fade by setting opacity to a tiny, but non-0, value to ensure
 * that it is not omitted by certain screen readers (ChromeVox specifically).
 */
export function accessiblyHide({
  $extraTransitionProperties = [],
  $hide = true,
  $transitionDurationSeconds = DEFAULT_HIDE_TRANSITION_DURATION,
}: AccessiblyHideProps): CSSObject {
  const properties = [...$extraTransitionProperties, 'opacity'].join(' ');

  return {
    opacity: $hide ? HIDDEN_OPACITY : 1,
    transition: `${properties} ${$transitionDurationSeconds}s ${HIDE_TIMING_FUNC}`,
  };
}

// istanbul ignore next: trivial
/**
 * Hides an element by setting opacity to a tiny, but non-0, value to ensure
 * that it is not omitted by certain screen readers (ChromeVox specifically).
 */
export function accessiblyHidden(): CSSObject {
  return {
    opacity: HIDDEN_OPACITY,
  };
}

export const HideableContainer = styled.div<{ $hide: boolean }>`
  ${accessiblyHide};
`;

const fadeInKeyFrames = keyframes`
  from {
    opacity: 0;
  }
  to {
    opacity: 1;
  }
`;

export type FadeInProps = {
  /**
   * Duration of the fade animation in seconds. Defaults to 0.5.
   */
  $durationSeconds?: number;
};

// istanbul ignore next: trivial
/**
 * A styled mixin to provide fade-in capabilities to styled components
 */
export function fadeIn({
  $durationSeconds = 0.5,
}: FadeInProps): FlattenSimpleInterpolation {
  return css`
    animation: ${fadeInKeyFrames} ${$durationSeconds}s;
  `;
}

// istanbul ignore next: trivial
/**
 * A component that will provide a fade-in animation to its children
 */
export const FadeInContainer = styled.div<FadeInProps>`
  ${fadeIn};
`;
