import type { FC } from 'react';
import { useEffect, useState } from 'react';
import { isBrowser, useDebounce, useEvent } from 'tachyon-utils';
import { WAND_ACTIVE_EVENT } from './WandActivityRoot';

// istanbul ignore next: trivial
function cursorStateChange(visibility: boolean): void {
  const event = new CustomEvent(WAND_ACTIVE_EVENT, {
    detail: { visibility },
  });
  document.dispatchEvent(event);
}

// istanbul ignore next: cost outweighs value
/**
 * Listens for mouse activity and fires LG's custom "cursorStateChange" to simulate wand behavior in dev mode.
 */
export const DevMouseAsWandObserver: FC = () => {
  // Disable hook linting because this is a stable if-check
  // (neither condition can change during runtime)
  /* eslint-disable react-hooks/rules-of-hooks */
  if (isBrowser() && process.env.DOCKER_PRODUCTION !== 'true') {
    const [active, setActive] = useState(false);
    const setActiveDebounced = useDebounce(setActive, 2000, { trailing: true });

    useEvent(
      'mousemove',
      () => {
        if (active) {
          setActiveDebounced(false);
        } else {
          setActive(true);
        }
      },
      window,
    );

    // Exit 'wand mode' when d-pad navigation occurs
    useEvent(
      'keydown',
      () => {
        if (active) {
          setActive(false);
        }
      },
      window,
    );

    useEffect(() => {
      cursorStateChange(active);
    }, [active]);
  }

  return null;
};
