import type { FC } from 'react';
import {
  createContext,
  useCallback,
  useContext,
  useEffect,
  useMemo,
  useState,
} from 'react';
import { useDebounce, useEffectOnce } from 'tachyon-utils';
import { RECENT_USER_INPUT_TIMEOUT_MS } from '../../../../config';

export type RecentUserInput = {
  cancelPersistedRecentUserInput: () => void;
  persistRecentUserInputIndefinitely: () => void;
  recentUserInput: boolean;
  simulateUserInput: () => void;
};

export const recentUserInputContext = createContext<RecentUserInput>({
  cancelPersistedRecentUserInput: () => null,
  persistRecentUserInputIndefinitely: () => null,
  recentUserInput: false,
  simulateUserInput: () => null,
});

export type UseRecentUserInput = {
  defaultActive?: boolean;
};

export function useRecentUserInput(
  props?: UseRecentUserInput,
): RecentUserInput {
  const recentUserInputContextInstance = useContext(recentUserInputContext);
  useEffectOnce(() => {
    if (props?.defaultActive) {
      recentUserInputContextInstance.simulateUserInput();
    }
  });
  return recentUserInputContextInstance;
}

export const RecentUserInputRoot: FC = ({ children }) => {
  const [recentUserInput, setRecentUserInput] = useState(false);
  const [persistedUserInput, setPersistedUserInput] = useState(false);

  const unsetRecentUserInput = useDebounce(
    () => {
      setRecentUserInput(false);
    },
    RECENT_USER_INPUT_TIMEOUT_MS,
    { trailing: true },
  );

  const simulateUserInput = useCallback(() => {
    // show controls if they are not already showing
    if (!recentUserInput) {
      setRecentUserInput(true);
    }
    // start or reset 'RECENT_USER_INPUT_TIMEOUT_MS' timer
    unsetRecentUserInput();
  }, [recentUserInput, unsetRecentUserInput]);

  useEffect(() => {
    window.addEventListener('keydown', simulateUserInput);

    return () => {
      window.removeEventListener('keydown', simulateUserInput);
    };
  }, [simulateUserInput]);

  const value = useMemo(
    () => ({
      cancelPersistedRecentUserInput: () => {
        setPersistedUserInput(false);
      },
      persistRecentUserInputIndefinitely: () => {
        setPersistedUserInput(true);
      },
      recentUserInput: persistedUserInput || recentUserInput,
      simulateUserInput,
    }),
    [persistedUserInput, recentUserInput, simulateUserInput],
  );

  return <recentUserInputContext.Provider children={children} value={value} />;
};

RecentUserInputRoot.displayName = 'RecentUserInputRoot';
