import { act, renderHook } from '@testing-library/react-hooks';
import { KeyValue } from 'tachyon-utils';
import { RECENT_USER_INPUT_TIMEOUT_MS } from '../../../../config';
import type { UseRecentUserInput } from '.';
import { RecentUserInputRoot, useRecentUserInput } from '.';

describe(RecentUserInputRoot, () => {
  function setup(overrides?: UseRecentUserInput) {
    return renderHook(() => useRecentUserInput(overrides), {
      wrapper: RecentUserInputRoot,
    });
  }

  beforeEach(() => {
    jest.resetAllMocks();
  });

  it("recentUserInput is true and then false after 'RECENT_USER_INPUT_TIMEOUT_MS' after a relevant key press", () => {
    const { result } = setup();
    const keyDown = new KeyboardEvent('keydown', { key: KeyValue.Down });

    expect(result.current.recentUserInput).toEqual(false);

    act(() => {
      window.dispatchEvent(keyDown);
    });
    expect(result.current.recentUserInput).toEqual(true);

    act(() => {
      jest.advanceTimersByTime(RECENT_USER_INPUT_TIMEOUT_MS);
    });
    expect(result.current.recentUserInput).toEqual(false);
  });

  it("recentUserInput remains true until 'RECENT_USER_INPUT_TIMEOUT_MS' elapses after a relevant key press", () => {
    const { result } = setup();
    const keyDown = new KeyboardEvent('keydown', { key: KeyValue.Down });

    expect(result.current.recentUserInput).toEqual(false);

    // stays open as long as relevant key presses are made
    for (let i = 0; i < 10; i++) {
      act(() => {
        jest.advanceTimersByTime(RECENT_USER_INPUT_TIMEOUT_MS / 2);
      });

      act(() => {
        window.dispatchEvent(keyDown);
      });
      expect(result.current.recentUserInput).toEqual(true);
    }

    // closes
    act(() => {
      jest.advanceTimersByTime(RECENT_USER_INPUT_TIMEOUT_MS);
    });
    expect(result.current.recentUserInput).toEqual(false);
  });

  it('recentUserInput remains true when persistRecentUserInputIndefinitely is called until cancelPersistedRecentUserInput is called', () => {
    const { result } = setup();
    expect(result.current.recentUserInput).toEqual(false);

    act(() => {
      result.current.persistRecentUserInputIndefinitely();
    });
    expect(result.current.recentUserInput).toEqual(true);

    act(() => {
      jest.advanceTimersByTime(RECENT_USER_INPUT_TIMEOUT_MS * 5);
    });
    expect(result.current.recentUserInput).toEqual(true);

    act(() => {
      result.current.cancelPersistedRecentUserInput();
    });
    expect(result.current.recentUserInput).toEqual(false);
  });

  it("recentUserInput is true when passed defaultActive: true until 'RECENT_USER_INPUT_TIMEOUT_MS' elapses", () => {
    const { result } = setup({ defaultActive: true });
    expect(result.current.recentUserInput).toEqual(true);
    act(() => {
      jest.advanceTimersByTime(RECENT_USER_INPUT_TIMEOUT_MS);
    });

    expect(result.current.recentUserInput).toEqual(false);
  });
});
