import type { FC } from 'react';
import { createContext, useState } from 'react';
import { createGlobalStyle } from 'styled-components';
import { useEffectOnce } from 'tachyon-utils';
import { useRecentUserInput } from './RecentUserInput';

export const wandActivityContext = createContext<boolean>(false);

type GlobalCursorDisableStyleProps = {
  disableCursor: boolean;
};

// Prevents the mouse from accidentally taking focus when not being used
export const GlobalCursorDisableStyle = createGlobalStyle<GlobalCursorDisableStyleProps>`
  body {
    cursor: ${({ disableCursor }) => disableCursor && 'none !important'};
  }
`;

/**
 * LG-specific handler for wand cursor visibility change. Adds
 * and removes the global hover class for Core-UI so that items
 * don't get stuck with hover styles when the cursor disappears
 * while an item is being hovered.
 *
 * http://webostv.developer.lge.com/develop/app-developer-guide/handling-system-ui-event/#cursor
 */
export const WAND_ACTIVE_EVENT = 'cursorStateChange';

// istanbul ignore next: cost outweighs value
export const WandActivityRoot: FC = ({ children }) => {
  const [wandActive, setWandActive] = useState(false);
  const { simulateUserInput } = useRecentUserInput();

  useEffectOnce(() => {
    function handler({ detail }: CursorStateChangeEvent): void {
      setWandActive(detail.visibility);
      if (detail.visibility) {
        simulateUserInput();
      }
    }

    document.addEventListener(WAND_ACTIVE_EVENT, handler);
    return () => {
      document.removeEventListener(WAND_ACTIVE_EVENT, handler);
    };
  });

  return (
    <wandActivityContext.Provider value={wandActive}>
      {children}
      <GlobalCursorDisableStyle disableCursor={!wandActive} />
    </wandActivityContext.Provider>
  );
};

WandActivityRoot.displayName = 'WandActivityRoot';
