import type { FC } from 'react';
import {
  createContext,
  useCallback,
  useContext,
  useMemo,
  useState,
} from 'react';
import { KeyValue } from 'tachyon-utils';
import { useSecretCode } from '../../../../hooks';
import { DebugInfoOverlay } from '../DebugInfoOverlay';
import { HoverOutlineDebugger } from '../HoverOutlineDebugger';
import { KeyDebugger } from '../KeyDebugger';

type StarshotVisualDebugModeRoot = {
  debugModeActive: boolean;
  toggleDebugMode: () => void;
};

export const defaultStarshotVisualDebugModeRootContext: StarshotVisualDebugModeRoot =
  {
    debugModeActive: false,
    toggleDebugMode: () => ({}),
  };

const starshotVisualDebugModeRootContext =
  createContext<StarshotVisualDebugModeRoot>(
    defaultStarshotVisualDebugModeRootContext,
  );

export const useStarshotVisualDebugMode = (): StarshotVisualDebugModeRoot =>
  useContext(starshotVisualDebugModeRootContext);

const debugSequence = [
  KeyValue.Right,
  KeyValue.Left,
  KeyValue.Left,
  KeyValue.Right,
  KeyValue.Right,
  KeyValue.Left,
  KeyValue.Right,
  KeyValue.Left,
  KeyValue.Left,
  KeyValue.Right,
  KeyValue.Right,
  KeyValue.Left,
  KeyValue.Down,
];

// istanbul ignore next: debug tools
export const StarshotVisualDebugModeRoot: FC = ({ children }) => {
  const [debugModeActive, setDebugModeActive] = useState(false);
  const toggleDebugMode = useCallback(() => {
    setDebugModeActive(!debugModeActive);
  }, [debugModeActive]);

  useSecretCode(debugSequence, toggleDebugMode);

  const ctx = useMemo(
    () => ({ debugModeActive, toggleDebugMode }),
    [debugModeActive, toggleDebugMode],
  );

  return (
    <starshotVisualDebugModeRootContext.Provider value={ctx}>
      {children}
      {debugModeActive && (
        <>
          <DebugInfoOverlay />
          <KeyDebugger />
          <HoverOutlineDebugger />
        </>
      )}
    </starshotVisualDebugModeRootContext.Provider>
  );
};

StarshotVisualDebugModeRoot.displayName = 'StarshotVisualDebugModeRoot';
