import { useRouter } from 'next/router';
import { parseUrl } from 'query-string';
import { useCurrentUser } from 'tachyon-auth';
import { defaultPageviewTracking, getCurrentUser } from 'tachyon-page-utils';
import {
  flattenHeaderOrParam,
  isBrowser,
  useConst,
  useEffectOnce,
} from 'tachyon-utils';
import { CONTEXT_PARAM } from '../../../config';
import {
  RouteName,
  linkPartsForRouteName,
  routeAndRouteParamsFromQueryString,
  routeNameFromRawPath,
} from '../../../routing';
import { getAccessAndRefreshTokenFromUrlFragment } from '../../../utils';
import { useChatAllowed } from '../../framework';
import type { StarshotPage } from '../types';

/**
 * Native app param to communicate to SST the IPC version. Useful for prompting user updates
 */
export const IPC_VERSION_PARAM = 'ipc-version';
/**
 * Native app param to set whether or not chat is allowed to be displayed to the user
 */
export const CHAT_ALLOWED_PARAM = 'chat-allowed';
/**
 * Param used to override the redirect target to a specific page instead of the home page
 * Ex: `destination=/hungry`
 * `destination=/hungry/home`
 */
export const DESTINATION_PARAM = 'destination';

const QueryParams = [IPC_VERSION_PARAM, CHAT_ALLOWED_PARAM, DESTINATION_PARAM];

export const AppShell: StarshotPage = () => {
  // use next router directly so we can push route strings instead of objects
  const router = useRouter();
  const { login, logout } = useCurrentUser();
  const { setIsChatAllowed } = useChatAllowed();
  // Capture the href on first render going down the tree so we don't lose it in
  // a useEffect race as part of URL cleanup (necessary because router-utils
  // doesn't capture fragments
  const href = useConst(() => (isBrowser() ? window.location.href : ''));

  useEffectOnce(() => {
    const { query } = parseUrl(href);
    // Iterate over each param key and set the cookie for each. The values stored
    // on the native app should be treated as the source of truth and an omission
    // of the value means the cookie should be cleared.
    QueryParams.forEach((key) => {
      // If the key is not present in the query no-op
      if (!(key in query)) {
        return;
      }
      const flattenedValue = flattenHeaderOrParam(query[key]);
      switch (key) {
        case IPC_VERSION_PARAM:
          // TODO: If IPC doesn't match we may need to serve different functionality
          break;
        case CHAT_ALLOWED_PARAM:
          if (flattenedValue) {
            setIsChatAllowed(flattenedValue !== 'false');
          }
          break;
      }
    });

    const authData = getAccessAndRefreshTokenFromUrlFragment(href);
    // if authData is defined but authToken is omitted call logout, otherwise call login
    if (authData) {
      if (authData.accessToken) {
        login({
          access_token: authData.accessToken,
          refresh_token: authData.refreshToken ?? '',
        });
      } else {
        logout();
      }
    }

    let path = '/';

    const encodedRedirectPath = flattenHeaderOrParam(query[DESTINATION_PARAM]);
    if (encodedRedirectPath) {
      const redirectPath = decodeURIComponent(encodedRedirectPath);
      // prevent attempting to navigate to invalid routes
      if (routeNameFromRawPath(redirectPath) !== RouteName.NotFound) {
        path = redirectPath;
      }
    }

    // Currently context is only used for redirecting back to a page after logging in
    // in the app. If context is present try to create a redirect route from it.
    const internalContext = flattenHeaderOrParam(query[CONTEXT_PARAM]);
    if (internalContext) {
      const routeData = routeAndRouteParamsFromQueryString(internalContext);

      if (routeData) {
        const { as } = linkPartsForRouteName(routeData);
        if (as) {
          path = as;
        }
      }
    }

    // We push this with the router so that users don't back out of the app from the home screen.
    router.push(path);
  });

  return null;
};

AppShell.navigationBehavior = () => ({});
AppShell.displayName = 'AppShell';
AppShell.currentUser = getCurrentUser;
AppShell.pageviewTracking = defaultPageviewTracking;
