import { useRouter } from 'next/router';
import { stringify } from 'query-string';
import { useCurrentUser } from 'tachyon-auth';
import { createMountWrapperFactory } from 'tachyon-test-utils';
import {
  ACCESS_TOKEN_FRAGMENT_PARAM,
  CONTEXT_PARAM,
  REFRESH_TOKEN_FRAGMENT_PARAM,
} from '../../../config';
import { useChatAllowed } from '../../framework';
import type { IPC_VERSION_PARAM } from '.';
import { AppShell, CHAT_ALLOWED_PARAM, DESTINATION_PARAM } from '.';

jest.mock('../../../routing', () => ({
  ...jest.requireActual('../../../routing'),
  Redirect: jest.fn(() => <div />),
}));

jest.mock('tachyon-auth', () => ({
  ...jest.requireActual('tachyon-auth'),
  useCurrentUser: jest.fn(),
}));

const mockUseCurrentUser = useCurrentUser as jest.Mock;
const mockLogin = jest.fn();
const mockLogout = jest.fn();

jest.mock('next/router', () => ({
  ...jest.requireActual('next/router'),
  useRouter: jest.fn(),
}));

const mockUseRouter = useRouter as jest.Mock;
const mockPush = jest.fn();

jest.mock('../../framework', () => ({
  ...jest.requireActual('../../framework'),
  useChatAllowed: jest.fn(),
}));

const mockUseChatAllowed = useChatAllowed as jest.Mock;
const mockSetIsChatAllowed = jest.fn();

describe(AppShell, () => {
  type Params = {
    [ACCESS_TOKEN_FRAGMENT_PARAM]?: string | null;
    [CHAT_ALLOWED_PARAM]?: string | null;
    [CONTEXT_PARAM]?: string | null;
    [DESTINATION_PARAM]?: string | null;
    [IPC_VERSION_PARAM]?: string | null;
    [REFRESH_TOKEN_FRAGMENT_PARAM]?: string | null;
  };

  const setup = (params?: Params) => {
    mockUseRouter.mockImplementation(() => ({
      push: mockPush,
    }));
    mockUseCurrentUser.mockImplementation(() => ({
      login: mockLogin,
      logout: mockLogout,
    }));

    mockUseChatAllowed.mockImplementation(() => ({
      setIsChatAllowed: mockSetIsChatAllowed,
    }));

    const testUrl = new URL('https://tv.twitch.tv/app-shell');

    const fragmentParams: Partial<Params> = {};
    for (const [key, value] of Object.entries(params ?? {})) {
      switch (key) {
        case ACCESS_TOKEN_FRAGMENT_PARAM:
        case REFRESH_TOKEN_FRAGMENT_PARAM:
          fragmentParams[key] = value;
          break;
        default:
          testUrl.searchParams.append(key, value ?? '');
          break;
      }
    }

    if (Object.keys(fragmentParams).length) {
      testUrl.hash = `#${stringify(fragmentParams)}`;
    }

    window.history.pushState({}, '', testUrl.toString());
    return createMountWrapperFactory(AppShell)();
  };

  describe('destination', () => {
    type RedirectTestCase = {
      expectedRedirect: string;
      providedRedirect: string | null;
    };

    it.each`
      expectedRedirect       | providedRedirect
      ${'/'}                 | ${''}
      ${'/'}                 | ${null}
      ${'/'}                 | ${'/this-is/not-a-route'}
      ${'/monstercat'}       | ${'/monstercat'}
      ${'/search?term=test'} | ${'/search?term%3Dtest'}
    `(
      'redirects to "$expectedRedirect" when DESTINATION_PARAM is "$providedRedirect"',
      ({ expectedRedirect, providedRedirect }: RedirectTestCase) => {
        setup({ [DESTINATION_PARAM]: providedRedirect });
        expect(mockPush).toHaveBeenCalledWith(expectedRedirect);
      },
    );
  });

  describe('context redirect', () => {
    type RedirectTestCase = {
      expectedRedirect: string;
      providedRedirect: string | null;
    };

    it.each`
      expectedRedirect      | providedRedirect
      ${'/'}                | ${''}
      ${'/'}                | ${null}
      ${'/'}                | ${'route%3Dthis-is-not-a-route'}
      ${'/monstercat'}      | ${'route%3Dchannel%26routeParams%3Dlogin%253Dmonstercat'}
      ${'/monstercat/home'} | ${'route%3Dchannel_home%26routeParams%3Dlogin%253Dmonstercat'}
    `(
      'redirects to "$expectedRedirect" when context redirect is "$providedRedirect"',
      ({ expectedRedirect, providedRedirect }: RedirectTestCase) => {
        setup({ [CONTEXT_PARAM]: providedRedirect });
        expect(mockPush).toHaveBeenCalledWith(expectedRedirect);
      },
    );
  });

  describe('chat allowed', () => {
    it('sets chat allowed to true when true is passed with the query param', () => {
      setup({ [CHAT_ALLOWED_PARAM]: 'true' });

      expect(mockSetIsChatAllowed).toHaveBeenCalledWith(true);
    });

    it('sets chat allowed to false when false is passed with the query param', () => {
      setup({ [CHAT_ALLOWED_PARAM]: 'false' });

      expect(mockSetIsChatAllowed).toHaveBeenCalledWith(false);
    });

    it('does not set chat allowed when the query param is omitted', () => {
      setup();

      expect(mockSetIsChatAllowed).not.toHaveBeenCalled();
    });
  });

  describe('login and logout', () => {
    it('login is called when an auth token is passed in', () => {
      setup({ [ACCESS_TOKEN_FRAGMENT_PARAM]: 'auth-test' });

      expect(mockLogin).toHaveBeenCalledTimes(1);
    });

    it('logout is called when the param is present but the value is empty', () => {
      setup({ [ACCESS_TOKEN_FRAGMENT_PARAM]: '' });

      expect(mockLogout).toHaveBeenCalledTimes(1);
    });

    it('neither login or logout are called when the param is omitted', () => {
      setup();

      expect(mockLogout).not.toHaveBeenCalled();
      expect(mockLogin).not.toHaveBeenCalled();
    });
  });
});
