import {
  setCaptionsCookie,
  useClosedCaptions,
  usePlaybackQuality,
} from 'pulsar';
import type { FC } from 'react';
import { useEffect } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import { HorizontalNav } from 'tachyon-tv-nav';
import { AlignItems, Display, JustifyContent, Layout } from 'twitch-core-ui';
import { useFollow } from '../../../../../hooks';
import { RouteName } from '../../../../../routing';
import {
  FocusableFollowButton,
  PlaybackInfo,
  PlaybackOverlayItem,
} from '../../../../common';
import { ClosedCaptionsButton, QualitySelectButton } from '../../../../player';
import { ChannelSettingsButton } from './ChannelSettingsButton';
import { ToggleChatButton } from './ToggleChatButton';
import type { BottomControls_channel } from './__generated__/BottomControls_channel.graphql';

export type BottomControlsProps = {
  channel: BottomControls_channel;
  chatVisible: boolean;
  focusIndex: number;
  onChatToggle: () => void;
};

export const BottomControlsBase: FC<BottomControlsProps> = ({
  channel,
  chatVisible,
  focusIndex,
  onChatToggle,
}) => {
  const { following } = useFollow(channel);
  const {
    available: captionsAvailable,
    enabled: captionsEnabled,
    toggle: toggleCaptions,
  } = useClosedCaptions();
  const qualitySettings = usePlaybackQuality();

  useEffect(() => {
    setCaptionsCookie(captionsEnabled);
  }, [captionsEnabled]);

  // Order matters for each of the buttons below
  const buttons = [
    <PlaybackOverlayItem key="settings">
      <ChannelSettingsButton focusIndex={1} login={channel.login} />
    </PlaybackOverlayItem>,
  ];

  // Only show the follow button if the channel isn't already followed to
  // avoid accidental unfollows.
  // This UX will be improved as part of: https://jira.xarth.tv/browse/EMP-3714
  if (!following) {
    buttons.push(
      <PlaybackOverlayItem key="follow">
        <FocusableFollowButton
          channel={channel}
          focusIndex={buttons.length + 1}
          iconOnly
          route={RouteName.Channel}
        />
      </PlaybackOverlayItem>,
    );
  }

  if (captionsAvailable) {
    buttons.push(
      <PlaybackOverlayItem key="captions">
        <ClosedCaptionsButton
          captionsEnabled={captionsEnabled}
          focusIndex={buttons.length + 1}
          onClick={toggleCaptions}
        />
      </PlaybackOverlayItem>,
    );
  }

  // Render this proactively even if the player hasn't initialized and qualities aren't yet
  // available for two reasons:
  // 1) Avoid a jarring UI update as a result of this being rendered after all of other UI has finished loading
  // 2) In a partial success case, there otherwise might not be anything to render
  //    in the bottom controls nav which can create a nav error state.
  buttons.push(
    <PlaybackOverlayItem key="playback-quality">
      <QualitySelectButton
        focusIndex={buttons.length + 1}
        qualitySettings={qualitySettings}
      />
    </PlaybackOverlayItem>,
  );

  // Not dynamic, pushed for consistency
  buttons.push(
    <PlaybackOverlayItem key="chat">
      <ToggleChatButton
        chatVisible={chatVisible}
        focusIndex={buttons.length + 1}
        onClick={onChatToggle}
      />
    </PlaybackOverlayItem>,
  );

  // profile link + buttons
  const elementCount = 1 + buttons.length;

  return (
    <HorizontalNav elementCount={elementCount} focusIndex={focusIndex}>
      <Layout
        display={Display.Flex}
        fullWidth
        justifyContent={JustifyContent.Between}
      >
        <PlaybackInfo
          channel={channel}
          focusIndex={0}
          gameDisplayName={channel.broadcastSettings?.game?.displayName}
          isLive
          recordedAt={channel.stream?.createdAt}
          title={channel.broadcastSettings?.title}
          viewCount={channel.stream?.viewersCount ?? 0}
        />
        <Layout
          alignItems={AlignItems.Center}
          display={Display.Flex}
          flexShrink={0}
        >
          {buttons}
        </Layout>
      </Layout>
    </HorizontalNav>
  );
};

BottomControlsBase.displayName = 'BottomControlsBase';

export const BottomControls = createFragmentContainer(BottomControlsBase, {
  channel: graphql`
    fragment BottomControls_channel on User {
      ...useFollow_targetChannel @relay(mask: false)
      ...FocusableFollowButton_channel
      ...PlaybackInfo_channel
      broadcastSettings {
        title
        game {
          displayName
        }
      }
      login
      stream {
        createdAt
        id
        viewersCount
      }
    }
  `,
});

BottomControls.displayName = 'BottomControls';
