import { datatype } from 'faker';
import { mockClosedCaptionsContext, setCaptionsCookie } from 'pulsar';
import { createMountWrapperFactory } from 'tachyon-test-utils';
import { HorizontalNav } from 'tachyon-tv-nav';
import { useFollow } from '../../../../../hooks';
import { FocusableFollowButton, PlaybackInfo } from '../../../../common';
import { ClosedCaptionsButton, QualitySelectButton } from '../../../../player';
import { ChannelSettingsButton } from './ChannelSettingsButton';
import { ToggleChatButton } from './ToggleChatButton';
import { BottomControlsBase } from '.';

jest.mock('../../../../common', () => ({
  ...jest.requireActual('../../../../common'),
  FocusableFollowButton: () => <div />,
  PlaybackInfo: () => <div />,
}));

jest.mock('pulsar', () => ({
  ...jest.requireActual('pulsar'),
  setCaptionsCookie: jest.fn(),
}));
const mockSetCaptionsCookie = setCaptionsCookie as jest.Mock;

jest.mock('../../../../../hooks', () => ({
  ...jest.requireActual('../../../../../hooks'),
  useFollow: jest.fn(),
}));
const mockFollow = useFollow as jest.Mock;

describe(BottomControlsBase, () => {
  const setup = createMountWrapperFactory(
    BottomControlsBase,
    () => ({
      channel: {
        ' $fragmentRefs': {
          FocusableFollowButton_channel: true,
          PlaybackInfo_channel: true,
        },
        ' $refType': 'BottomControls_channel',
        broadcastSettings: {
          game: {
            displayName: 'Game displayname',
          },
          title: 'Stream title',
        },
        id: datatype.uuid(),
        login: 'therealderekt',
        self: null,
        stream: {
          createdAt: '2021-03-30T11:39:13Z',
          id: datatype.uuid(),
          viewersCount: datatype.number(),
        },
      },
      chatVisible: true,
      currentUser: {
        ' $refType': 'BottomControls_currentUser',
        id: datatype.uuid(),
      },
      focusIndex: 0,
      onChatToggle: jest.fn(),
      visible: true,
    }),
    {
      wrappingContexts: () => [
        mockClosedCaptionsContext({
          available: false,
          enabled: false,
          subscribe: jest.fn(),
          toggle: jest.fn(),
        }),
      ],
    },
  );

  beforeEach(() => {
    mockSetCaptionsCookie.mockReset();
    mockFollow.mockReset();
    mockFollow.mockReturnValue({ following: false });
  });

  it('does not render the captions button when unavailable', () => {
    const { wrapper } = setup();
    expect(wrapper.find(ClosedCaptionsButton)).not.toExist();
  });

  it('renders the captions button when available', () => {
    const { wrapper } = setup(undefined, {
      wrappingContexts: [
        mockClosedCaptionsContext({
          available: true,
        }),
      ],
    });
    expect(wrapper.find(ClosedCaptionsButton)).toExist();
  });

  it('invokes onToggleChat when the chat button is clicked', () => {
    const { props, wrapper } = setup();
    expect(props.onChatToggle).not.toHaveBeenCalled();

    wrapper.find(ToggleChatButton).prop('onClick')();
    expect(props.onChatToggle).toHaveBeenCalledTimes(1);
  });

  it('does not render the follow button if the user is already following', () => {
    mockFollow.mockReturnValue({ following: true });
    const { wrapper } = setup();
    expect(wrapper.find(FocusableFollowButton)).not.toExist();
  });

  it('always renders the channel settings button', () => {
    const { wrapper } = setup();
    expect(wrapper.find(ChannelSettingsButton)).toExist();
  });

  it('assigns expected focusIndex values when all interactables are rendered', () => {
    mockFollow.mockReturnValue({ following: false });
    const { wrapper } = setup(undefined, {
      wrappingContexts: [
        mockClosedCaptionsContext({
          available: true,
        }),
      ],
    });

    expect(wrapper.find(HorizontalNav)).toHaveProp({ elementCount: 6 });
    expect(wrapper.find(PlaybackInfo)).toHaveProp({ focusIndex: 0 });
    expect(wrapper.find(ChannelSettingsButton)).toHaveProp({
      focusIndex: 1,
    });
    expect(wrapper.find(FocusableFollowButton)).toHaveProp({ focusIndex: 2 });
    expect(wrapper.find(ClosedCaptionsButton)).toHaveProp({ focusIndex: 3 });
    expect(wrapper.find(QualitySelectButton)).toHaveProp({ focusIndex: 4 });
    expect(wrapper.find(ToggleChatButton)).toHaveProp({ focusIndex: 5 });
  });
});
