import type { FC } from 'react';
import { useState } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import styled from 'styled-components';
import { HorizontalNav, VerticalNav } from 'tachyon-tv-nav';
import { throttle, useConstCallback } from 'tachyon-utils';
import { DarkThemeMap } from 'twitch-core-ui-tokens';
import { CLIENT_ID } from '../../../../config';
import { useUpdateVideoPlaybackPosition } from '../../../../hooks';
import { Chat } from '../../../Chat';
import { PlayingUILayout } from '../../../common';
import type { StreamPlayerProps } from '../../../player';
import { StreamPlayer } from '../../../player';
import { BottomControls } from './BottomControls';
import type { ChannelPlayingUI_channel } from './__generated__/ChannelPlayingUI_channel.graphql';
import type { ChannelPlayingUI_currentUser } from './__generated__/ChannelPlayingUI_currentUser.graphql';

type ChannelPlayingUIProps = Pick<StreamPlayerProps, 'streamToken'> & {
  channel: ChannelPlayingUI_channel;
  currentUser: ChannelPlayingUI_currentUser | null;
  focusIndex: number;
};

export const CHAT_WIDTH_VW = 25;
export const PLAYER_WIDTH_WITH_CHAT_VW = 100 - CHAT_WIDTH_VW;

type ChatLayoutProps = {
  $displayChat: boolean;
};

export const ScPlayerSection = styled.section<ChatLayoutProps>`
  height: 100%;
  position: relative;
  width: ${({ $displayChat }) =>
    $displayChat ? PLAYER_WIDTH_WITH_CHAT_VW : 100}vw;
`;

export const ScChatAside = styled.aside<ChatLayoutProps>`
  background: ${DarkThemeMap['color-background-base']};
  height: 100%;
  ${({ $displayChat }) =>
    $displayChat ? '' : `transform: translateX(${CHAT_WIDTH_VW}vw);`};
  position: absolute;
  right: 0;
  top: 0;
  width: ${CHAT_WIDTH_VW}vw;
`;

export const ChannelPlayingUIBase: FC<ChannelPlayingUIProps> = ({
  channel,
  currentUser,
  focusIndex,
  streamToken,
}) => {
  const [displayChat, setDisplayChat] = useState(true);
  const toggleChat = useConstCallback(
    // This throttle time is set pretty high because people shouldn't be
    // intending to toggle this on and off a lot in succession.
    throttle(
      () => {
        setDisplayChat((prevState) => !prevState);
      },
      500,
      {
        leading: true,
      },
    ),
  );

  useUpdateVideoPlaybackPosition({
    userID: currentUser?.id,
    videoID: channel.stream?.archiveVideo?.id,
    videoType: 'LIVE',
  });

  return (
    <HorizontalNav elementCount={displayChat ? 2 : 1} focusIndex={focusIndex}>
      <VerticalNav elementCount={2} focusIndex={0} initialChildFocusIndex={1}>
        <ScPlayerSection $displayChat={displayChat}>
          <PlayingUILayout
            channel={channel}
            live
            player={
              <StreamPlayer
                autoPlay
                channel={channel}
                currentUser={currentUser}
                isPreviewPlayer={false}
                streamToken={streamToken}
              />
            }
          >
            <PlayingUILayout.BackButton focusIndex={0} />
            <PlayingUILayout.Controls>
              <BottomControls
                channel={channel}
                chatVisible={displayChat}
                focusIndex={1}
                onChatToggle={toggleChat}
              />
            </PlayingUILayout.Controls>
          </PlayingUILayout>
        </ScPlayerSection>
      </VerticalNav>
      <ScChatAside $displayChat={displayChat}>
        <Chat channel={channel} clientApiId={CLIENT_ID} focusIndex={1} />
      </ScChatAside>
    </HorizontalNav>
  );
};

ChannelPlayingUIBase.displayName = 'ChannelPlayingUIBase';

export const ChannelPlayingUI = createFragmentContainer(ChannelPlayingUIBase, {
  channel: graphql`
    fragment ChannelPlayingUI_channel on User {
      stream {
        archiveVideo {
          id
        }
      }
      id
      login
      displayName
      roles {
        isAffiliate
        isPartner
      }
      ...StreamPlayer_channel
      ...BottomControls_channel
    }
  `,
  currentUser: graphql`
    fragment ChannelPlayingUI_currentUser on User {
      ...StreamPlayer_currentUser
      ...types_currentUser @relay(mask: false)
    }
  `,
});
