import {
  ChannelPlaybackRestrictionsRoot,
  PlayerControllerRoot,
  PlayerType,
} from 'pulsar';
import { graphql } from 'react-relay/hooks';
import { Platform, useStaticEnvironment } from 'tachyon-environment';
import type {
  ChannelPathParameters,
  ChannelInitialProps as DefaultChannelInitialProps,
} from 'tachyon-page-utils';
import {
  channelIsFound,
  channelIsNotFoundServerside,
  channelPathGetInitialProps,
  channelWatchPageviewTracking,
  getCurrentUser,
  isPlayingChannel,
} from 'tachyon-page-utils';
import { isBrowser } from 'tachyon-utils';
import { Redirect, RouteName } from '../../../routing';
import { StarshotMain } from '../../common';
import { NotFoundError } from '../../errors';
import { ChannelRestriction } from '../../player';
import type { StarshotPage } from '../types';
import { ChannelPlayingUI } from './ChannelPlayingUI';
import type {
  Channel_QueryResponse,
  Channel_QueryVariables,
} from './__generated__/Channel_Query.graphql';

export type ChannelInitialProps =
  DefaultChannelInitialProps<Channel_QueryVariables>;

export type ChannelProps = Channel_QueryResponse & ChannelInitialProps;

export const Channel: StarshotPage<
  ChannelInitialProps,
  ChannelProps,
  ChannelPathParameters
> = ({ channel, currentUser, queryVariables, ...streamToken }) => {
  const {
    common: { platform },
  } = useStaticEnvironment();

  if (!channelIsFound(channel)) {
    return <NotFoundError />;
  }

  if (!isPlayingChannel(channel)) {
    return (
      <Redirect
        params={{
          route: RouteName.ChannelHome,
          routeParams: { login: queryVariables.login },
        }}
        replace
      />
    );
  }

  if (platform === Platform.Switch) {
    return (
      <Redirect
        params={{
          route: RouteName.ChannelInterstitial,
          routeParams: { login: queryVariables.login },
        }}
        replace
      />
    );
  }

  return (
    <StarshotMain>
      <PlayerControllerRoot>
        <ChannelPlaybackRestrictionsRoot channel={channel}>
          <ChannelRestriction channel={channel} focusIndex={0}>
            <ChannelPlayingUI
              channel={channel}
              currentUser={currentUser}
              focusIndex={0}
              streamToken={streamToken}
            />
          </ChannelRestriction>
        </ChannelPlaybackRestrictionsRoot>
      </PlayerControllerRoot>
    </StarshotMain>
  );
};

Channel.currentUser = getCurrentUser;
Channel.displayName = 'Channel';
Channel.isNotFoundServerside = channelIsNotFoundServerside;
Channel.navigationBehavior = () => ({});
Channel.pageviewTracking = channelWatchPageviewTracking;
Channel.getInitialProps = (context) =>
  channelPathGetInitialProps(context, {
    platform: context.platform,
    playerType: PlayerType.Pulsar,
    skipPlayToken: !isBrowser(),
  });

Channel.query = graphql`
  query Channel_Query(
    $login: String!
    $platform: String!
    $playerType: String!
    $skipPlayToken: Boolean!
  ) {
    channel: user(login: $login) {
      ...ChannelPlayingUI_channel
      ...ChannelRestriction_channel @relay(mask: false)
      id
      login
      stream {
        id
        game {
          name
        }
      }
    }
    currentUser {
      ...ChannelPlayingUI_currentUser
      ...types_currentUser @relay(mask: false)
    }
    ...StreamPlayer_token
  }
`;
