import { internet } from 'faker';
import { Platform, useStaticEnvironment } from 'tachyon-environment';
import { validId } from 'tachyon-relay';
import { createShallowWrapperFactory, randomId } from 'tachyon-test-utils';
import { Redirect, RouteName } from '../../../routing';
import { NotFoundError } from '../../errors';
import { Channel } from '.';

jest.mock('../../Chat/chat.worker', () => jest.fn());
jest.mock('tachyon-environment', () => ({
  ...jest.requireActual('tachyon-environment'),
  useStaticEnvironment: jest.fn(),
}));

const mockUseStaticEnvironment = useStaticEnvironment as jest.Mock;

describe(Channel, () => {
  mockUseStaticEnvironment.mockReturnValue({
    common: { platform: Platform.StarshotDev },
  });
  const setup = createShallowWrapperFactory(Channel, () => ({
    ' $fragmentRefs': { StreamPlayer_token: true },
    channel: {
      ' $fragmentRefs': {
        ChannelPlayingUI_channel: true,
      },
      broadcastSettings: {
        isMature: false,
      },
      displayName: internet.userName(),
      id: validId(randomId()),
      login: internet.userName(),
      stream: {
        game: {
          name: internet.userName(),
        },
        id: validId(randomId()),
        restrictionType: null,
        self: {
          canWatch: true,
        },
      },
    },
    currentUser: null,
    queryVariables: {
      login: internet.userName(),
      platform: Platform.StarshotDev,
      playerType: 'pulsar',
      skipPlayToken: true,
    },
  }));

  it('redirects to the interstitial when the platform is switch', () => {
    mockUseStaticEnvironment.mockReturnValueOnce({
      common: { platform: Platform.Switch },
    });
    const { wrapper } = setup({
      queryVariables: { login: 'foo' },
    });

    expect(wrapper.find(Redirect)).toHaveProp({
      params: {
        route: RouteName.ChannelInterstitial,
        routeParams: { login: 'foo' },
      },
      replace: true,
    });
  });

  it('redirects to ChannelHome when the channel is offline', () => {
    const { wrapper } = setup({
      channel: { stream: null },
      queryVariables: { login: 'foo' },
    });

    expect(wrapper.find(Redirect)).toHaveProp({
      params: {
        route: RouteName.ChannelHome,
        routeParams: { login: 'foo' },
      },
      replace: true,
    });
  });

  it('renders the not found UX when channel is not found', () => {
    const { wrapper } = setup({ channel: null });
    expect(wrapper.find(NotFoundError)).toExist();
  });
});
