import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import { getStreamType } from 'tachyon-more-ui';
import { getPlayingChannel } from 'tachyon-page-utils';
import { HorizontalNav } from 'tachyon-tv-nav';
import { reduceToNonNullNodes } from 'tachyon-utils';
import { CoreButtonSize, Display, FlexWrap, Layout } from 'twitch-core-ui';
import { RouteName } from '../../../../../routing';
import {
  FocusableFollowButton,
  FocusableReportButton,
} from '../../../../common';
import { WatchLatestVodButton } from './WatchLatestVodButton';
import { WatchStreamButton } from './WatchStreamButton';
import type { BannerButtonsRow_channel } from './__generated__/BannerButtonsRow_channel.graphql';

export type BannerButtonsRowBaseProps = {
  channel: BannerButtonsRow_channel;
  focusIndex: number;
};

export const BannerButtonsRowBase: FC<BannerButtonsRowBaseProps> = ({
  channel,
  focusIndex,
}) => {
  const streamType = getStreamType({
    broadcaster: channel,
    type: channel.stream?.type ?? null,
  });
  const playingChannel = getPlayingChannel(channel);
  const recentVideos = reduceToNonNullNodes(channel.videos?.edges);

  let firstButton = null;
  if (streamType && playingChannel) {
    firstButton = (
      <Layout padding={{ bottom: 0.5, right: 0.5 }}>
        <WatchStreamButton
          focusIndex={0}
          login={playingChannel.login}
          type={streamType}
        />
      </Layout>
    );
  } else {
    if (recentVideos.length > 0) {
      firstButton = (
        <Layout padding={{ bottom: 0.5, right: 0.5 }}>
          <WatchLatestVodButton focusIndex={0} videoId={recentVideos[0].id} />
        </Layout>
      );
    }
  }

  // If the stream is offline, there's no watch/hosting CTA button,
  // so we expect only 2 elements if offline
  return (
    <HorizontalNav elementCount={firstButton ? 3 : 2} focusIndex={focusIndex}>
      <Layout
        display={Display.Flex}
        flexWrap={FlexWrap.Wrap}
        margin={{ top: 2 }}
      >
        {firstButton}
        <Layout padding={{ bottom: 0.5 }}>
          <FocusableFollowButton
            channel={channel}
            focusIndex={1 - (firstButton ? 0 : 1)}
            route={RouteName.ChannelHome}
            size={CoreButtonSize.Small}
            withUnfocusBackground
          />
        </Layout>
        <Layout padding={{ bottom: 0.5, left: 0.5 }}>
          <FocusableReportButton
            focusIndex={2 - (firstButton ? 0 : 1)}
            login={channel.login}
            withUnfocusBackground
          />
        </Layout>
      </Layout>
    </HorizontalNav>
  );
};
BannerButtonsRowBase.displayName = 'BannerButtonsRowBase';

// istanbul ignore next: trivial
export const BannerButtonsRow = createFragmentContainer(BannerButtonsRowBase, {
  channel: graphql`
    fragment BannerButtonsRow_channel on User {
      ...FocusableFollowButton_channel
      displayName
      hosting {
        displayName
        id
        login
        stream {
          id
          type
        }
      }
      id
      login
      stream {
        id
        type
      }
      videos(first: 1, sort: TIME) {
        edges {
          node {
            id
          }
        }
      }
    }
  `,
});
BannerButtonsRow.displayName = 'BannerButtonsRow';
