import { internet } from 'faker';
import { StreamType } from 'tachyon-more-ui';
import { validId } from 'tachyon-relay';
import { createShallowWrapperFactory, randomId } from 'tachyon-test-utils';
import { HorizontalNav } from 'tachyon-tv-nav';
import {
  FocusableFollowButton,
  FocusableReportButton,
} from '../../../../common';
import { WatchLatestVodButton } from './WatchLatestVodButton';
import { WatchStreamButton } from './WatchStreamButton';
import { BannerButtonsRowBase } from '.';

jest.mock('tachyon-tv-nav', () => ({
  ...jest.requireActual('tachyon-tv-nav'),
  useFocus: jest.fn(() => ({ focused: true })),
}));

const mockLiveChannel = (type?: string) =>
  ({
    ' $fragmentRefs': {
      FocusableFollowButton_channel: true,
    },
    ' $refType': 'BannerButtonsRow_channel',
    displayName: internet.userName(),
    hosting: null,
    id: validId(randomId()),
    login: internet.userName(),
    stream: {
      id: validId(randomId()),
      type: type ?? 'live',
    },
  } as const);

const mockHostingChannel = () =>
  ({
    ' $fragmentRefs': {
      FocusableFollowButton_channel: true,
    },
    ' $refType': 'BannerButtonsRow_channel',
    displayName: internet.userName(),
    hosting: {
      displayName: internet.userName(),
      id: validId(randomId()),
      login: internet.userName(),
      stream: {
        id: validId(randomId()),
        type: 'hosting',
      },
    },
    id: validId(randomId()),
    login: internet.userName(),
    stream: null,
  } as const);

const mockOfflineChannel = (
  videos: {
    edges: [
      {
        node: {
          id: string;
        };
      },
    ];
  } | null = null,
) =>
  ({
    ' $fragmentRefs': {
      FocusableFollowButton_channel: true,
    },
    ' $refType': 'BannerButtonsRow_channel',
    displayName: internet.userName(),
    hosting: null,
    id: validId(randomId()),
    login: internet.userName(),
    stream: null,
    videos,
  } as const);

describe(BannerButtonsRowBase, () => {
  const setup = createShallowWrapperFactory(BannerButtonsRowBase, () => ({
    channel: mockOfflineChannel(),
    focusIndex: 0,
  }));

  describe('live', () => {
    it('contains watch/follow/report buttons in the horizontal nav when expanded', () => {
      const { wrapper } = setup({
        channel: mockLiveChannel('live'),
      });
      expect(wrapper.find(HorizontalNav)).toExist();
      expect(wrapper.find(HorizontalNav).prop('elementCount')).toEqual(3);
      expect(wrapper.find(WatchStreamButton)).toExist();
      expect(wrapper.find(FocusableFollowButton)).toExist();
      expect(wrapper.find(FocusableReportButton)).toExist();
    });

    it('passes the correct stream type prop to the watch button', () => {
      const { wrapper } = setup({
        channel: mockLiveChannel('live'),
      });
      expect(wrapper.find(WatchStreamButton).prop('type')).toEqual(
        StreamType.Live,
      );
    });
  });

  describe('hosting', () => {
    it('contains watch/follow/report buttons in the horizontal nav when expanded', () => {
      const { wrapper } = setup({
        channel: mockHostingChannel(),
      });
      expect(wrapper.find(HorizontalNav)).toExist();
      expect(wrapper.find(HorizontalNav).prop('elementCount')).toEqual(3);
      expect(wrapper.find(WatchStreamButton)).toExist();
      expect(wrapper.find(FocusableFollowButton)).toExist();
      expect(wrapper.find(FocusableReportButton)).toExist();
    });

    it('passes correct the stream type prop to the watch button', () => {
      const { wrapper } = setup({
        channel: mockHostingChannel(),
      });
      expect(wrapper.find(WatchStreamButton).prop('type')).toEqual(
        StreamType.Hosting,
      );
    });
  });

  describe('rerun', () => {
    it('contains watch/follow/report buttons in the horizontal nav when expanded', () => {
      const { wrapper } = setup({
        channel: mockLiveChannel('rerun'),
      });
      expect(wrapper.find(HorizontalNav)).toExist();
      expect(wrapper.find(HorizontalNav).prop('elementCount')).toEqual(3);
      expect(wrapper.find(WatchStreamButton)).toExist();
      expect(wrapper.find(FocusableFollowButton)).toExist();
      expect(wrapper.find(FocusableReportButton)).toExist();
    });

    it('passes correct the stream type prop to the watch button', () => {
      const { wrapper } = setup({
        channel: mockLiveChannel('rerun'),
      });
      expect(wrapper.find(WatchStreamButton).prop('type')).toEqual(
        StreamType.Rerun,
      );
    });
  });

  describe('offline', () => {
    it('contains follow/report buttons in the horizontal nav when expanded and no vods are present', () => {
      const { wrapper } = setup({
        channel: mockOfflineChannel(),
      });
      expect(wrapper.find(HorizontalNav)).toExist();
      expect(wrapper.find(HorizontalNav).prop('elementCount')).toEqual(2);
      expect(wrapper.find(FocusableFollowButton)).toExist();
      expect(wrapper.find(WatchLatestVodButton)).not.toExist();
      expect(wrapper.find(FocusableReportButton)).toExist();
    });

    it('contains watch/follow/report buttons in the horizontal nav when expanded and there are vods present', () => {
      const { wrapper } = setup({
        channel: mockOfflineChannel({
          edges: [
            {
              node: {
                id: validId(randomId()),
              },
            },
          ],
        }),
      });
      expect(wrapper.find(HorizontalNav)).toExist();
      expect(wrapper.find(HorizontalNav).prop('elementCount')).toEqual(3);
      expect(wrapper.find(WatchLatestVodButton)).toExist();
      expect(wrapper.find(FocusableFollowButton)).toExist();
      expect(wrapper.find(FocusableReportButton)).toExist();
    });
  });
});
