import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import styled from 'styled-components';
import { useIntl } from 'tachyon-intl';
import { StreamType, getStreamType } from 'tachyon-more-ui';
import { getPlayingChannel } from 'tachyon-page-utils';
import { getFormattedUserDisplayName } from 'tachyon-utils';
import { CoreText } from 'twitch-core-ui';
import { DarkThemeMap } from 'twitch-core-ui-tokens';
import type { BannerChannelStatus_channel } from './__generated__/BannerChannelStatus_channel.graphql';

// istanbul ignore next: trivial
const ScStatusText = styled(CoreText)<{ $textColor: string }>`
  color: ${(props) => props.$textColor};
  font-weight: 500;
`;

export type BannerChannelStatusBaseProps = {
  channel: BannerChannelStatus_channel;
};

/**
 * A component that renders the status of a channel on the profile banner.
 * Handles all the complexities of determining the channel/hosting status
 * to simplify the main banner component.
 */
export const BannerChannelStatusBase: FC<BannerChannelStatusBaseProps> = ({
  channel,
}) => {
  const { formatMessage } = useIntl();

  const playingChannel = getPlayingChannel(channel);

  const streamType = getStreamType({
    broadcaster: channel,
    type: channel.stream?.type ?? null,
  });

  if (!playingChannel || !streamType) {
    return (
      <ScStatusText
        $textColor={DarkThemeMap['color-text-alt-2']}
        children={formatMessage('OFFLINE', 'ProfileBanner')}
      />
    );
  }
  switch (streamType) {
    case StreamType.Live:
    case StreamType.Premiere:
      return (
        <ScStatusText
          $textColor={DarkThemeMap['color-fill-alert']}
          children={formatMessage('LIVE NOW', 'ProfileBanner')}
        />
      );
    case StreamType.Hosting:
      return (
        <ScStatusText
          $textColor={DarkThemeMap['color-fill-warn']}
          children={formatMessage(
            'HOSTING {name}',
            { name: getFormattedUserDisplayName(playingChannel) },
            'ProfileBanner',
          )}
        />
      );
    case StreamType.Rerun:
      return (
        <ScStatusText
          $textColor={DarkThemeMap['color-text-alt-2']}
          children={formatMessage('RERUN', 'ProfileBanner')}
        />
      );
    default:
      const exhaust: never = streamType;
      return exhaust;
  }
};
BannerChannelStatusBase.displayName = 'BannerChannelStatusBase';

// istanbul ignore next: trivial
export const BannerChannelStatus = createFragmentContainer(
  BannerChannelStatusBase,
  {
    channel: graphql`
      fragment BannerChannelStatus_channel on User {
        displayName
        hosting {
          displayName
          id
          login
          stream {
            id
            type
          }
        }
        id
        login
        stream {
          id
          type
        }
      }
    `,
  },
);
BannerChannelStatus.displayName = 'BannerChannelStatus';
