import type { FC } from 'react';
import { useState } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import styled from 'styled-components';
import { useIntl } from 'tachyon-intl';
import { useFocus } from 'tachyon-tv-nav';
import { getFormattedUserDisplayName } from 'tachyon-utils';
import {
  Avatar,
  CoreText,
  Display,
  FlexDirection,
  FontSize,
  FontWeight,
  JustifyContent,
  Layout,
  Title,
  TitleSize,
} from 'twitch-core-ui';
import { DarkThemeMap } from 'twitch-core-ui-tokens';
import { HideableContainer } from '../../../common';
import type { ProfilePlayerBaseProps } from '../ProfilePlayer';
import { ProfilePlayer } from '../ProfilePlayer';
import { BannerButtonsRow } from './BannerButtonsRow';
import { BannerChannelStatus } from './BannerChannelStatus';
import { BannerStatsLine } from './BannerStatsLine';
import type { ProfileBanner_channel } from './__generated__/ProfileBanner_channel.graphql';
import type { ProfileBanner_currentUser } from './__generated__/ProfileBanner_currentUser.graphql';

// istanbul ignore next: trivial
const ScProfileBannerBg = styled(Layout)`
  background-color: ${DarkThemeMap['color-background-body']};
`;

type ScProfileBannerLayoutProps = {
  $expanded: boolean;
};

/**
 * When we expand the banner from a closed state, we want to immediately animate
 * the height of the banner while delaying the fade in of the expanded elements
 * so that there isn't any awkward overlap between the expanded banner content
 * and the shelf content as the shelves animate.
 *
 * When compressing the banner again, the reverse is true. We immediately fade
 * out expanded banner elements while delaying the height animation.
 */
export const IMMEDIATE_ANIMATION = '200ms ease-out';
export const DELAYED_ANIMATION = '100ms ease-out 100ms';

export const EXPANDED_HEIGHT_REM = '38rem';
export const COMPRESSED_HEIGHT_REM = '25rem';

export const ScExpandableHeightContainer = styled.div<ScProfileBannerLayoutProps>`
  height: ${(props) =>
    props.$expanded ? EXPANDED_HEIGHT_REM : COMPRESSED_HEIGHT_REM};
  ${({ $expanded }) => !$expanded && 'overflow: hidden;'};
  transition: height
    ${(props) => (props.$expanded ? IMMEDIATE_ANIMATION : DELAYED_ANIMATION)};
`;

// istanbul ignore next: trivial
const ScProfileBannerLayout = styled(Layout)`
  opacity: 1;
  padding-top: 14rem;
`;

// istanbul ignore next: trivial
const ScChannelInfoLayout = styled(Layout)`
  width: 42rem;
`;

export type ProfileBannerProps = Pick<
  ProfilePlayerBaseProps,
  'streamToken' | 'vodPreviewToken'
> & {
  channel: ProfileBanner_channel;
  currentUser: ProfileBanner_currentUser | null;
  focusIndex: number;
  forceExpanded: boolean;
};

export const ProfileBannerBase: FC<ProfileBannerProps> = ({
  channel,
  currentUser,
  focusIndex,
  forceExpanded,
  streamToken,
  vodPreviewToken,
}) => {
  const intl = useIntl();
  const { formatMessage } = intl;
  const { focused } = useFocus(focusIndex);
  const [hovered, setHovered] = useState(false);

  const name = getFormattedUserDisplayName(channel);
  // We want to expand the banner if any of the following conditions are true:
  // 1. an element within the banner has focus
  // 2. a sibling higher on the page is focused, in which case the parent will pass a force expanded prop
  // 3. the mouse/wand is hovering over the div containing the banner content, but the lower shelf content has focus
  const showExpandedContent = focused || forceExpanded || hovered;

  return (
    <ScProfileBannerBg>
      <ScExpandableHeightContainer
        $expanded={showExpandedContent}
        onMouseEnter={() => {
          setHovered(true);
        }}
        onMouseLeave={() => {
          setHovered(false);
        }}
      >
        <ScProfileBannerLayout
          display={Display.Flex}
          fullWidth
          justifyContent={JustifyContent.Between}
          padding={{ bottom: 4, x: 5 }}
        >
          <Layout display={Display.Flex} flexDirection={FlexDirection.Row}>
            <Avatar
              alt=""
              size={64}
              src={channel.profileImageURL}
              userLogin={channel.login}
            />
            <Layout
              display={Display.Flex}
              flexDirection={FlexDirection.Column}
              margin={{ left: 2 }}
            >
              <BannerChannelStatus channel={channel} />
              <Title size={TitleSize.Small}>{name}</Title>
              <BannerStatsLine
                followers={channel.followers?.totalCount}
                views={channel.profileViewCount}
              />
              <HideableContainer $hide={!showExpandedContent}>
                <ScChannelInfoLayout
                  display={Display.Flex}
                  flexDirection={FlexDirection.Column}
                  margin={{ top: 2 }}
                >
                  {channel.description && channel.description.length > 0 && (
                    <>
                      <CoreText
                        fontSize={FontSize.Size8}
                        fontWeight={FontWeight.SemiBold}
                      >
                        {formatMessage(
                          'About {broadcasterName}:',
                          { broadcasterName: name },
                          'ProfileBanner',
                        )}
                      </CoreText>
                      <Layout padding={{ top: 0.5 }}>
                        <CoreText fontSize={FontSize.Size8}>
                          {channel.description}
                        </CoreText>
                      </Layout>
                    </>
                  )}
                  <BannerButtonsRow channel={channel} focusIndex={focusIndex} />
                </ScChannelInfoLayout>
              </HideableContainer>
            </Layout>
          </Layout>
          <HideableContainer $hide={!showExpandedContent}>
            <ProfilePlayer
              channel={channel}
              currentUser={currentUser}
              streamToken={streamToken}
              vodPreviewToken={vodPreviewToken}
            />
          </HideableContainer>
        </ScProfileBannerLayout>
      </ScExpandableHeightContainer>
    </ScProfileBannerBg>
  );
};

ProfileBannerBase.displayName = 'ProfileBannerBase';

// istanbul ignore next: trivial
export const ProfileBanner = createFragmentContainer(ProfileBannerBase, {
  channel: graphql`
    fragment ProfileBanner_channel on User {
      ...BannerButtonsRow_channel
      ...BannerChannelStatus_channel
      ...ProfilePlayer_channel
      description
      displayName
      followers {
        totalCount
      }
      hosting {
        id
        login
        profileImageURL(width: 70)
      }
      id
      login
      profileImageURL(width: 70)
      profileViewCount
    }
  `,
  currentUser: graphql`
    fragment ProfileBanner_currentUser on User {
      ...ProfilePlayer_currentUser
    }
  `,
});

ProfileBanner.displayName = 'ProfileBanner';
