import { datatype, internet, lorem } from 'faker';
import { validId } from 'tachyon-relay';
import { createShallowWrapperFactory, randomId } from 'tachyon-test-utils';
import { useFocus } from 'tachyon-tv-nav';
import { HideableContainer } from '../../../common';
import {
  COMPRESSED_HEIGHT_REM,
  DELAYED_ANIMATION,
  EXPANDED_HEIGHT_REM,
  IMMEDIATE_ANIMATION,
  ProfileBannerBase,
  ScExpandableHeightContainer,
} from '.';

jest.mock('tachyon-tv-nav', () => ({
  ...jest.requireActual('tachyon-tv-nav'),
  useFocus: jest.fn(() => ({ focused: true })),
}));

const mockUseFocus = useFocus as jest.Mock;

const mockLiveChannel = (type?: string) =>
  ({
    ' $fragmentRefs': {
      BannerButtonsRow_channel: true,
      BannerChannelStatus_channel: true,
      ProfilePlayer_channel: true,
    },
    ' $refType': 'ProfileBanner_channel',
    description: lorem.words(),
    displayName: internet.userName(),
    followers: {
      totalCount: datatype.number(),
    },
    hosting: null,
    id: validId(randomId()),
    login: internet.userName(),
    profileImageURL: internet.url(),
    profileViewCount: datatype.number(),
    stream: {
      id: validId(randomId()),
      previewImageURL: internet.url(),
      type: type ?? 'live',
      viewersCount: datatype.number(),
    },
  } as const);

const mockCurrentUser = () =>
  ({
    ' $fragmentRefs': {
      ProfilePlayer_currentUser: true,
    },
    ' $refType': 'ProfileBanner_currentUser',
  } as const);

describe(ProfileBannerBase, () => {
  const setup = createShallowWrapperFactory(ProfileBannerBase, () => ({
    channel: mockLiveChannel(),
    currentUser: mockCurrentUser(),
    focusIndex: 0,
    forceExpanded: true,
    streamToken: {
      ' $fragmentRefs': { StreamPlayer_token: true },
    },
    vodPreviewToken: {
      ' $fragmentRefs': { VodPreviewPlayerWrapper_previewToken: true },
    },
  }));

  describe('render', () => {
    it('propagates expanded=false to expandable components when not focused and not force-expanded', () => {
      mockUseFocus.mockReturnValue({ focused: false });
      const { wrapper } = setup({
        channel: mockLiveChannel(),
        forceExpanded: false,
      });
      expect(wrapper.find(ScExpandableHeightContainer)).toHaveProp(
        '$expanded',
        false,
      );
      wrapper.find(HideableContainer).forEach((component) => {
        expect(component).toHaveProp('$hide', true);
      });
    });

    it('propagates expanded=true to expandable components when focused', () => {
      mockUseFocus.mockReturnValue({ focused: true });
      const { wrapper } = setup({
        channel: mockLiveChannel(),
        forceExpanded: false,
      });
      expect(wrapper.find(ScExpandableHeightContainer)).toHaveProp(
        '$expanded',
        true,
      );
      wrapper.find(HideableContainer).forEach((component) => {
        expect(component).toHaveProp('$hide', false);
      });
    });

    it('propagates expanded=true to expandable components when force-expanded', () => {
      mockUseFocus.mockReturnValue({ focused: false });
      const { wrapper } = setup({
        channel: mockLiveChannel(),
        forceExpanded: true,
      });
      expect(wrapper.find(ScExpandableHeightContainer)).toHaveProp(
        '$expanded',
        true,
      );
      wrapper.find(HideableContainer).forEach((component) => {
        expect(component).toHaveProp('$hide', false);
      });
    });

    it('expands the banner when hovered and not focused or force-expanded, un-expands when no longer hovered', () => {
      mockUseFocus.mockReturnValue({
        focused: false,
      });
      const { wrapper } = setup({
        channel: mockLiveChannel(),
        forceExpanded: false,
      });

      expect(wrapper.find(ScExpandableHeightContainer).prop('$expanded')).toBe(
        false,
      );

      wrapper.find(ScExpandableHeightContainer).prop('onMouseEnter')();
      expect(wrapper.find(ScExpandableHeightContainer).prop('$expanded')).toBe(
        true,
      );

      wrapper.find(ScExpandableHeightContainer).prop('onMouseLeave')();
      expect(wrapper.find(ScExpandableHeightContainer).prop('$expanded')).toBe(
        false,
      );
    });
  });
});

describe('ScExpandableHeightContainer', () => {
  const setup = createShallowWrapperFactory(
    ScExpandableHeightContainer,
    () => ({
      $expanded: true,
    }),
  );

  it('is taller when expanded', () => {
    const { wrapper } = setup({ $expanded: true });
    wrapper.setProps({ $expanded: true });
    expect(wrapper).toHaveStyleRule('height', EXPANDED_HEIGHT_REM);
    expect(wrapper).toHaveStyleRule(
      'transition',
      `height ${IMMEDIATE_ANIMATION}`,
    );
  });

  it('is shorter when not expanded', () => {
    const { wrapper } = setup({ $expanded: false });
    wrapper.setProps({ $expanded: false });
    expect(wrapper).toHaveStyleRule('height', COMPRESSED_HEIGHT_REM);
    expect(wrapper).toHaveStyleRule(
      'transition',
      `height ${DELAYED_ANIMATION}`,
    );
  });

  it('allows overflow when expanded', () => {
    const { wrapper } = setup({ $expanded: true });
    wrapper.setProps({ $expanded: true });
    expect(wrapper).not.toHaveStyleRule('overflow');
  });

  it('hides overflow when not expanded', () => {
    const { wrapper } = setup({ $expanded: false });
    wrapper.setProps({ $expanded: false });
    expect(wrapper).toHaveStyleRule('overflow', 'hidden');
  });
});
