import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import styled from 'styled-components';
import { useIntl } from 'tachyon-intl';
import { getPlayingChannel } from 'tachyon-page-utils';
import { reduceToNonNullNodes } from 'tachyon-utils';
import {
  AlignItems,
  Background,
  CoreText,
  Display,
  JustifyContent,
  Layout,
  Position,
} from 'twitch-core-ui';
import type {
  StreamPreviewPlayerProps,
  VodPreviewPlayerBaseProps,
} from '../../../player';
import { StreamPreviewPlayer, VodPreviewPlayer } from '../../../player';
import type { ProfilePlayer_channel } from './__generated__/ProfilePlayer_channel.graphql';
import type { ProfilePlayer_currentUser } from './__generated__/ProfilePlayer_currentUser.graphql';

// istanbul ignore next: trivial
const ScPlayerContainer = styled(Layout)`
  height: 19.6875rem;
  width: 35rem;
`;

export type ProfilePlayerBaseProps = Pick<
  StreamPreviewPlayerProps,
  'streamToken'
> &
  Pick<VodPreviewPlayerBaseProps, 'vodPreviewToken'> & {
    channel: ProfilePlayer_channel;
    currentUser: ProfilePlayer_currentUser | null;
  };

export const ProfilePlayerBase: FC<ProfilePlayerBaseProps> = ({
  channel,
  currentUser,
  streamToken,
  vodPreviewToken,
}) => {
  const { formatMessage } = useIntl();
  const playingChannel = getPlayingChannel(channel);
  const videos = reduceToNonNullNodes(channel.videos?.edges);
  const mostRecentVideo = videos.length > 0 && videos[0];

  return (
    <ScPlayerContainer
      background={Background.Base}
      position={Position.Relative}
    >
      {playingChannel ? (
        <StreamPreviewPlayer
          channel={channel}
          currentUser={currentUser}
          streamToken={streamToken}
        />
      ) : mostRecentVideo ? (
        <VodPreviewPlayer
          currentUser={currentUser}
          video={mostRecentVideo}
          vodPreviewToken={vodPreviewToken}
        />
      ) : (
        <Layout
          alignItems={AlignItems.Center}
          display={Display.Flex}
          fullHeight
          fullWidth
          justifyContent={JustifyContent.Center}
        >
          <CoreText>
            {formatMessage('No content available', 'ProfilePlayerBase')}
          </CoreText>
        </Layout>
      )}
    </ScPlayerContainer>
  );
};
ProfilePlayerBase.displayName = 'ProfilePlayerBase';

// Consistent fragment for structure of the top level user and hosted user
// eslint-disable-next-line no-unused-expressions
graphql`
  fragment ProfilePlayer_channelBase on User {
    id
    login
    stream {
      id
      type
      viewersCount
    }
  }
`;

// istanbul ignore next: trivial
export const ProfilePlayer = createFragmentContainer(ProfilePlayerBase, {
  channel: graphql`
    fragment ProfilePlayer_channel on User {
      ...StreamPreviewPlayer_channel
      hosting {
        ...ProfilePlayer_channelBase @relay(mask: false)
      }
      ...ProfilePlayer_channelBase @relay(mask: false)
      videos(first: 1, sort: TIME) {
        edges {
          node {
            id
            ...VodPreviewPlayer_video
          }
        }
      }
    }
  `,
  currentUser: graphql`
    fragment ProfilePlayer_currentUser on User {
      ...StreamPreviewPlayer_currentUser
      ...VodPreviewPlayer_currentUser
    }
  `,
});
ProfilePlayer.displayName = 'ProfilePlayer';
