import { PlayerType } from 'pulsar';
import type { FC, ReactElement } from 'react';
import { graphql } from 'react-relay/hooks';
import type {
  ChannelPathParameters,
  ChannelInitialProps as DefaultChannelInitialProps,
} from 'tachyon-page-utils';
import {
  channelHomePageviewTracking,
  channelIsFound,
  channelIsNotFoundServerside,
  channelPathGetInitialProps,
  getCurrentUser,
} from 'tachyon-page-utils';
import { VerticalNav, useFocus } from 'tachyon-tv-nav';
import { isBrowser } from 'tachyon-utils';
import { WATCHABLE_CARD_CONFIG } from '../../../config';
import {
  HorizontalShelf,
  StarshotMain,
  VerticalShelfList,
  VerticalShelvesPage,
} from '../../common';
import { NotFoundError } from '../../errors';
import type { StarshotPage } from '../types';
import { NoVideoShelvesMessage } from './NoVideoShelvesMessage';
import type { ProfileBannerProps } from './ProfileBanner';
import { ProfileBanner } from './ProfileBanner';
import { VideoItemRenderer } from './VideoItemRenderer';
import type {
  ChannelHome_QueryResponse,
  ChannelHome_QueryVariables,
} from './__generated__/ChannelHome_Query.graphql';
import type { VideoShelf, VodOrClipCard } from './utils';
import { getVideoShelves } from './utils';

function shelfItemRenderer(
  node: VodOrClipCard,
  focusIndex: number,
): ReactElement {
  return (
    <VideoItemRenderer focusIndex={focusIndex} key={focusIndex} node={node} />
  );
}

function shelfRenderer(shelf: VideoShelf, focusIndex: number): ReactElement {
  return (
    <HorizontalShelf
      config={WATCHABLE_CARD_CONFIG}
      focusIndex={focusIndex}
      itemRenderer={shelfItemRenderer}
      items={shelf.cards}
      key={shelf.id}
      title={shelf.title}
      virtualized
    />
  );
}

type ProfilePageContentProps = Pick<
  ProfileBannerProps,
  'streamToken' | 'vodPreviewToken'
> & {
  channel: NonNullable<ChannelHome_QueryResponse['channel']>;
  currentUser: ChannelHome_QueryResponse['currentUser'];
};

/**
 * A component that holds all the content of the profile page
 *
 * We want to force the profile banner to render its expanded view if the
 * top nav is focused in addition to if the banner is focused, so we gather that
 * information about the top nav focus here and pass it down to the banner.
 *
 * The easiest way to achieve this is for this outer component to know if it
 * has focus. If it doesn't, we assume the top nav has focus, so we should
 * expand the profile banner.
 *
 * This needs to be its own outer component, otherwise the `useFocus` will read
 * the root level `VerticalNav` rather than the one provided by `StarshotMain`
 */
export const ProfilePageContent: FC<ProfilePageContentProps> = ({
  channel,
  currentUser,
  streamToken,
  vodPreviewToken,
}) => {
  const { focused } = useFocus(0);

  const renderableShelves = getVideoShelves(channel.videoShelves);
  const hasShelves = renderableShelves.length > 0;

  return (
    <VerticalNav
      elementCount={hasShelves ? 2 : 1}
      focusIndex={0}
      takeFocusOnFirstRender
    >
      <VerticalShelvesPage>
        <VerticalShelvesPage.ElevatedContent>
          <ProfileBanner
            channel={channel}
            currentUser={currentUser}
            focusIndex={0}
            forceExpanded={!focused}
            streamToken={streamToken}
            vodPreviewToken={vodPreviewToken}
          />
        </VerticalShelvesPage.ElevatedContent>
        <VerticalShelvesPage.ScrollUnderContent>
          {hasShelves ? (
            <VerticalShelfList
              focusIndex={1}
              items={renderableShelves}
              shelfRenderer={shelfRenderer}
            />
          ) : (
            // If no shelves, display a "no content" message rather than an error
            <NoVideoShelvesMessage />
          )}
        </VerticalShelvesPage.ScrollUnderContent>
      </VerticalShelvesPage>
    </VerticalNav>
  );
};
ProfilePageContent.displayName = 'ProfilePageContent';

export type ChannelHomeInitialProps =
  DefaultChannelInitialProps<ChannelHome_QueryVariables>;

export type ChannelHomeProps = ChannelHome_QueryResponse &
  ChannelHomeInitialProps;

export const ChannelHome: StarshotPage<
  ChannelHomeInitialProps,
  ChannelHomeProps,
  ChannelPathParameters
> = ({ channel, currentUser, queryVariables, ...fragmentRefs }) => {
  // If bad channel, just do normal "not found" behavior
  if (!channelIsFound(channel)) {
    return <NotFoundError />;
  }

  return (
    <StarshotMain>
      <ProfilePageContent
        channel={channel}
        currentUser={currentUser}
        streamToken={fragmentRefs}
        vodPreviewToken={fragmentRefs}
      />
    </StarshotMain>
  );
};

ChannelHome.currentUser = getCurrentUser;
ChannelHome.displayName = 'ChannelHome';
ChannelHome.isNotFoundServerside = channelIsNotFoundServerside;
ChannelHome.pageviewTracking = channelHomePageviewTracking;
ChannelHome.getInitialProps = (context) =>
  channelPathGetInitialProps(context, {
    platform: context.platform,
    playerType: PlayerType.Max480pNoAds,
    skipPlayToken: !isBrowser(),
  });
ChannelHome.navigationBehavior = () => ({
  displayBackButton: true,
  displayNavMenu: true,
});

// eslint-disable-next-line no-unused-expressions
graphql`
  fragment ChannelHome_card on VideoShelfItem {
    __typename
    ... on Clip {
      ...FocusableClipCard_clip
    }
    ... on Video {
      ...FocusableVodCard_video
    }
  }
`;

// eslint-disable-next-line no-unused-expressions
graphql`
  fragment ChannelHome_shelf on VideoShelf {
    id
    title
  }
`;

// eslint-disable-next-line no-unused-expressions
graphql`
  fragment ChannelHome_shelves on VideoShelfConnection {
    edges {
      node {
        ...ChannelHome_shelf @relay(mask: false)
        items {
          ...ChannelHome_card @relay(mask: false)
        }
      }
    }
  }
`;

ChannelHome.query = graphql`
  query ChannelHome_Query(
    $login: String!
    $platform: String!
    $playerType: String!
    $skipPlayToken: Boolean!
  ) {
    channel: user(login: $login) {
      id
      login
      stream {
        id
      }
      videoShelves {
        ...ChannelHome_shelves @relay(mask: false)
      }
      ...ProfileBanner_channel
    }
    currentUser {
      ...ProfileBanner_currentUser
      ...types_currentUser @relay(mask: false)
    }
    ...StreamPlayer_token
    ...VodPreviewPlayerWrapper_previewToken
  }
`;
