import { datatype, date, internet, lorem } from 'faker';
import { Platform } from 'tachyon-environment';
import { validId } from 'tachyon-relay';
import { createShallowWrapperFactory, randomId } from 'tachyon-test-utils';
import { VerticalNav, useFocus } from 'tachyon-tv-nav';
import { mockRouter } from '../../../routing/test-mocks';
import { VerticalShelfList } from '../../common';
import { NotFoundError } from '../../errors';
import { NoVideoShelvesMessage } from './NoVideoShelvesMessage';
import { ProfileBanner } from './ProfileBanner';
import type { VideoShelf } from './utils';
import { ChannelHome, ProfilePageContent } from '.';

jest.mock('tachyon-tv-nav', () => ({
  ...jest.requireActual('tachyon-tv-nav'),
  useFocus: jest.fn(() => ({ focused: true })),
}));

const mockUseFocus = useFocus as jest.Mock;

const mockUseCurrentUser = { loggedIn: true };
jest.mock('tachyon-auth', () => ({
  ...jest.requireActual('tachyon-auth'),
  useCurrentUser: () => mockUseCurrentUser,
}));

const mockAppEnvironment = { appEnvironment: 'development' };
jest.mock('tachyon-environment', () => ({
  ...jest.requireActual('tachyon-environment'),
  useStaticEnvironment: () => ({
    common: mockAppEnvironment,
  }),
}));

jest.mock('../../../routing/useTachyonRouter', () => ({
  useTachyonRouter: () => mockRouter,
}));

jest.mock('../../common', () => {
  const { MockHorizontalShelf } = jest.requireActual(
    '../../common/HorizontalShelf/test-mocks',
  );
  const { MockVerticalShelfList } = jest.requireActual(
    '../../common/VerticalShelfList/test-mocks',
  );
  return {
    ...jest.requireActual('../../common'),
    FocusableClipCard: jest.fn(() => <div />),
    FocusableVodCard: jest.fn(() => <div />),
    HorizontalShelf: MockHorizontalShelf,
    VerticalShelfList: MockVerticalShelfList,
  };
});

const mockVideoCardQuery = () =>
  ({
    ' $fragmentRefs': {
      FocusableVodCard_video: true,
    },
    __typename: 'Video',
    createdAt: date.past(),
    game: {
      boxArtURL: internet.url(),
      displayName: lorem.word(),
    },
    id: validId(randomId()),
    lengthSeconds: datatype.number({ min: 0 }),
    previewThumbnailURL: internet.url(),
    title: lorem.word(),
    viewCount: datatype.number({ min: 0 }),
  } as const);

const mockClipCardQuery = () =>
  ({
    ' $fragmentRefs': {
      FocusableClipCard_clip: true,
    },
    __typename: 'Clip',
    createdAt: date.past(),
    durationSeconds: datatype.number({ min: 0 }),
    game: {
      boxArtURL: internet.url(),
      displayName: lorem.word(),
    },
    id: validId(randomId()),
    thumbnailURL: internet.url(),
    title: lorem.word(),
    viewCount: datatype.number({ min: 0 }),
  } as const);

const mockVideoShelfQuery = () =>
  ({
    id: validId(randomId()),
    items: [mockVideoCardQuery(), mockVideoCardQuery()],
    title: lorem.word(),
  } as const);

const mockClipShelfQuery = () =>
  ({
    id: validId(randomId()),
    items: [mockClipCardQuery(), mockClipCardQuery()],
    title: lorem.word(),
  } as const);

describe(ChannelHome, () => {
  const setup = createShallowWrapperFactory(ChannelHome, () => ({
    ' $fragmentRefs': {
      StreamPlayer_token: true,
      VodPreviewPlayerWrapper_previewToken: true,
    },
    channel: {
      ' $fragmentRefs': {
        ProfileBanner_channel: true,
      },
      id: datatype.uuid(),
      login: internet.userName(),
      stream: null,
      videoShelves: null,
    },
    currentUser: {
      ' $fragmentRefs': {
        ProfileBanner_currentUser: true,
      },
      id: validId(randomId()),
      login: internet.userName(),
      roles: {
        isStaff: false,
      },
    },
    queryVariables: {
      login: internet.userName(),
      platform: Platform.StarshotDev,
      playerType: 'pulsar',
      skipPlayToken: true,
    },
  }));

  beforeEach(() => {
    mockAppEnvironment.appEnvironment = 'development';
  });

  describe('render', () => {
    it('renders not found on missing channel', () => {
      const { wrapper } = setup({ channel: null });
      expect(wrapper.find(NotFoundError)).toExist();
    });
  });
});

describe(ProfilePageContent, () => {
  const setup = createShallowWrapperFactory(ProfilePageContent, () => ({
    channel: {
      ' $fragmentRefs': {
        ProfileBanner_channel: true,
      },
      id: datatype.uuid(),
      login: internet.userName(),
      stream: null,
      videoShelves: null,
    },
    currentUser: {
      ' $fragmentRefs': {
        ProfileBanner_currentUser: true,
      },
      id: validId(randomId()),
      login: internet.userName(),
      roles: {
        isStaff: false,
      },
    },
    queryVariables: {
      login: internet.userName(),
    },
    streamToken: {
      ' $fragmentRefs': { StreamPlayer_token: true },
    },
    vodPreviewToken: {
      ' $fragmentRefs': { VodPreviewPlayerWrapper_previewToken: true },
    },
  }));

  beforeEach(() => {
    mockAppEnvironment.appEnvironment = 'development';
  });

  describe('render', () => {
    it('renders no content message if there are no shelves', () => {
      const { wrapper } = setup({ channel: { videoShelves: { edges: [] } } });
      expect(wrapper.find(NoVideoShelvesMessage)).toExist();
    });

    it('renders no content message if there is a single shelf with null items', () => {
      const { wrapper } = setup({
        channel: {
          videoShelves: {
            edges: [
              {
                node: {
                  id: validId(randomId()),
                  items: null,
                  title: lorem.word(),
                },
              },
            ],
          },
        },
      });
      expect(wrapper.find(NoVideoShelvesMessage)).toExist();
    });

    it('has only one focusable element if there is no content', () => {
      const { wrapper } = setup({ channel: { videoShelves: { edges: [] } } });
      expect(wrapper.find(VerticalNav).props().elementCount).toEqual(1);
    });
  });

  describe('banner', () => {
    it('propagates focus-related props correctly to the banner component', () => {
      // Focusing the banner implies that we should not be force-expanding the banner
      mockUseFocus.mockReturnValue({ focused: true });
      const mockShelf = mockVideoShelfQuery();
      const { wrapper } = setup({
        channel: {
          videoShelves: {
            edges: [
              {
                node: mockShelf,
              },
            ],
          },
        },
      });

      const banner = wrapper.find(ProfileBanner);
      expect(banner).toExist();

      const bannerProps = banner.props();
      expect(bannerProps.focusIndex).toEqual(0);
      expect(bannerProps.forceExpanded).toEqual(false);
      expect(bannerProps.channel).not.toBeNull();
    });
  });

  describe('shelves', () => {
    it('propagates title props correctly to child shelves', () => {
      const { wrapper } = setup({
        channel: {
          videoShelves: {
            edges: [
              {
                node: {
                  id: '1234',
                  items: [mockVideoCardQuery(), mockVideoCardQuery()],
                  title: 'Foo',
                },
              },
              {
                node: {
                  id: '1234',
                  items: [mockClipCardQuery(), mockClipCardQuery()],
                  title: 'Bar',
                },
              },
            ],
          },
        },
      });

      const items = wrapper.find(VerticalShelfList).prop<VideoShelf[]>('items');
      expect(items).toHaveLength(2);
      expect(items[0].title).toEqual('Foo');
      expect(items[1].title).toEqual('Bar');
    });

    it('filters and propagates a shelf containing video elements', () => {
      const mockShelf = mockVideoShelfQuery();
      const { wrapper } = setup({
        channel: {
          videoShelves: {
            edges: [
              {
                node: mockShelf,
              },
            ],
          },
        },
      });

      const items = wrapper.find(VerticalShelfList).prop<VideoShelf[]>('items');
      expect(items[0].cards).toHaveLength(2);
    });

    it('filters and propagates a shelf containing clip elements', () => {
      const mockShelf = mockClipShelfQuery();
      const { wrapper } = setup({
        channel: {
          videoShelves: {
            edges: [
              {
                node: mockShelf,
              },
            ],
          },
        },
      });

      const items = wrapper.find(VerticalShelfList).prop<VideoShelf[]>('items');
      expect(items[0].cards).toHaveLength(2);
    });
  });

  it('contains two focusable elements if shelves are present', () => {
    const { wrapper } = setup({
      channel: {
        videoShelves: {
          edges: [
            {
              node: {
                id: '1234',
                items: [mockVideoCardQuery(), mockVideoCardQuery()],
                title: 'Foo',
              },
            },
          ],
        },
      },
    });

    expect(wrapper.find(VerticalNav).props().elementCount).toEqual(2);
  });
});
