import type { FC } from 'react';
import { graphql, useFragment } from 'react-relay/hooks';
import { useIntl } from 'tachyon-intl';
import { formatLiveViewersCount, getBoxArtSrcAndSrcSet } from 'tachyon-more-ui';
import {
  CoreImage,
  CoreText,
  Display,
  FlexDirection,
  FontWeight,
  JustifyContent,
  Layout,
  TextType,
} from 'twitch-core-ui';
import type { StreamDetails_channel$key } from './__generated__/StreamDetails_channel.graphql';

export const streamDetailsFragment = graphql`
  fragment StreamDetails_channel on User {
    broadcastSettings {
      game {
        boxArtURL
        displayName
      }
      title
    }
    stream {
      viewersCount
    }
  }
`;

export type StreamDetailsProps = {
  channel: StreamDetails_channel$key;
};

// istanbul ignore next: trivial
export const StreamDetails: FC<StreamDetailsProps> = ({
  channel: channelRef,
}) => {
  const channel = useFragment(streamDetailsFragment, channelRef);
  const { formatMessage, formatNumberShort } = useIntl();
  return (
    <Layout display={Display.Flex} flexDirection={FlexDirection.Row}>
      <Layout margin={{ right: 2 }}>
        <CoreImage
          alt=""
          sizes={[{ size: `7rem` }]}
          {...getBoxArtSrcAndSrcSet(channel.broadcastSettings?.game?.boxArtURL)}
        />
      </Layout>
      <Layout
        display={Display.Flex}
        flexDirection={FlexDirection.Column}
        justifyContent={JustifyContent.Center}
      >
        <Layout margin={{ bottom: 0.5 }}>
          <CoreText fontWeight={FontWeight.SemiBold} type={TextType.H6}>
            {channel.broadcastSettings?.title}
          </CoreText>
        </Layout>
        <Layout margin={{ bottom: 0.5 }}>
          <CoreText type={TextType.H6}>
            {channel.broadcastSettings?.game?.displayName}
          </CoreText>
        </Layout>
        {channel.stream?.viewersCount && (
          <Layout margin={{ bottom: 0.5 }}>
            <CoreText type={TextType.H6}>
              {formatLiveViewersCount(
                formatMessage,
                channel.stream.viewersCount,
                formatNumberShort(channel.stream.viewersCount, {
                  maximumSignificantDigits: 2,
                }),
              )}
            </CoreText>
          </Layout>
        )}
      </Layout>
    </Layout>
  );
};

StreamDetails.displayName = 'InterstitialStreamDetails';
