import {
  ChannelPlaybackRestrictionsRoot,
  PlayerControllerRoot,
  PlayerType,
} from 'pulsar';
import { useMemo } from 'react';
import { graphql } from 'react-relay/hooks';
import { Platform, useStaticEnvironment } from 'tachyon-environment';
import { StreamType, getStreamType } from 'tachyon-more-ui';
import type {
  ChannelPathParameters,
  ChannelInitialProps as DefaultChannelInitialProps,
} from 'tachyon-page-utils';
import {
  channelIsFound,
  channelPathGetInitialProps,
  defaultPageviewTracking,
  getCurrentUser,
  getPlayingChannel,
} from 'tachyon-page-utils';
import { isBrowser } from 'tachyon-utils';
import { CLIENT_ID } from '../../../config';
import {
  useInterstitialPlayAttempt,
  useIsContentPlayableOnSwitch,
} from '../../../hooks';
import { Redirect, RouteName } from '../../../routing';
import { Chat } from '../../Chat';
import { InterstitialLayout, StarshotMain } from '../../common';
import { NotFoundError } from '../../errors';
import { useChatAllowed } from '../../framework';
import { ChannelRestriction, StreamPlayer } from '../../player';
import type { StarshotPage } from '../types';
import { StreamDetails } from './StreamDetails';
import type {
  ChannelInterstitial_QueryResponse,
  ChannelInterstitial_QueryVariables,
} from './__generated__/ChannelInterstitial_Query.graphql';

export type ChannelInterstitialInitialProps =
  DefaultChannelInitialProps<ChannelInterstitial_QueryVariables>;

export type ChannelInterstitialProps = ChannelInterstitial_QueryResponse &
  ChannelInterstitialInitialProps;

export const ChannelInterstitial: StarshotPage<
  ChannelInterstitialInitialProps,
  ChannelInterstitialProps,
  ChannelPathParameters
> = ({ channel, currentUser, queryVariables, ...streamToken }) => {
  const { attemptedPlay, resetAttemptedPlay, triggerAttemptedPlay } =
    useInterstitialPlayAttempt();
  const {
    common: { platform },
  } = useStaticEnvironment();
  const { isChatAllowed } = useChatAllowed();
  const { isContentPlayable, setManifestUrl } = useIsContentPlayableOnSwitch();
  const chat = useMemo(
    () =>
      channel && (
        <Chat channel={channel} clientApiId={CLIENT_ID} focusIndex={0} />
      ),
    [channel],
  );

  if (!channelIsFound(channel)) {
    return <NotFoundError />;
  }

  const playingChannel = getPlayingChannel(channel);

  const streamType = getStreamType({
    broadcaster: channel,
    type: channel.stream?.type ?? null,
  });

  const shouldRedirect =
    !playingChannel ||
    !streamType ||
    streamType === StreamType.Hosting ||
    platform !== Platform.Switch;

  if (shouldRedirect) {
    return (
      <Redirect
        params={{
          route: RouteName.ChannelHome,
          routeParams: { login: queryVariables.login },
        }}
        replace
      />
    );
  }

  return (
    <StarshotMain>
      <PlayerControllerRoot>
        <ChannelPlaybackRestrictionsRoot channel={channel}>
          <ChannelRestriction
            channel={channel}
            focusIndex={0}
            passThrough={!attemptedPlay}
          >
            <InterstitialLayout
              attemptedPlay={attemptedPlay}
              channel={channel}
              chat={isChatAllowed ? chat : null}
              contentType="stream"
              isChannelLive={streamType === StreamType.Live}
              isContentPlayable={isContentPlayable}
              player={
                <StreamPlayer
                  autoPlay={false}
                  channel={channel}
                  currentUser={currentUser}
                  onManifestUrlCreated={setManifestUrl}
                  streamToken={streamToken}
                />
              }
              resetAttemptedPlay={resetAttemptedPlay}
              triggerPlayAttempt={triggerAttemptedPlay}
            >
              <StreamDetails channel={channel} />
            </InterstitialLayout>
          </ChannelRestriction>
        </ChannelPlaybackRestrictionsRoot>
      </PlayerControllerRoot>
    </StarshotMain>
  );
};

ChannelInterstitial.currentUser = getCurrentUser;
ChannelInterstitial.displayName = 'ChannelInterstitial';
ChannelInterstitial.navigationBehavior = () => ({
  displayNavMenu: true,
});
ChannelInterstitial.pageviewTracking = defaultPageviewTracking;
ChannelInterstitial.getInitialProps = (context) =>
  channelPathGetInitialProps(context, {
    platform: context.platform,
    playerType: PlayerType.Pulsar,
    skipPlayToken: !isBrowser(),
  });

ChannelInterstitial.query = graphql`
  query ChannelInterstitial_Query(
    $login: String!
    $platform: String!
    $playerType: String!
    $skipPlayToken: Boolean!
  ) {
    channel: user(login: $login) {
      ...InterstitialLayout_channel
      ...StreamDetails_channel
      ...StreamPlayer_channel
      ...ChannelRestriction_channel @relay(mask: false)
      displayName
      hosting {
        displayName
        id
        login
        stream {
          id
          type
        }
      }
      login
      id
      stream {
        id
        type
      }
    }
    currentUser {
      ...StreamPlayer_currentUser
      ...types_currentUser @relay(mask: false)
    }
    ...StreamPlayer_token
  }
`;
