import { datatype, internet } from 'faker';
import {
  Platform,
  mockCommonDevelopmentEnvironment,
  useStaticEnvironment,
} from 'tachyon-environment';
import { StreamType } from 'tachyon-more-ui';
import { validId } from 'tachyon-relay';
import { createShallowWrapperFactory, randomId } from 'tachyon-test-utils';
import { Redirect } from '../../../routing';
import { InterstitialLayout } from '../../common';
import { NotFoundError } from '../../errors';
import { useChatAllowed } from '../../framework';
import { ChannelInterstitial } from '.';

jest.mock('tachyon-environment', () => ({
  ...jest.requireActual('tachyon-environment'),
  useStaticEnvironment: jest.fn(),
}));

jest.mock('../../Chat/chat.worker', () => jest.fn());

jest.mock('../../framework', () => ({
  useChatAllowed: jest.fn(),
}));

const mockUseStaticEnvironment = useStaticEnvironment as jest.Mock;
const mockUseChatAllowed = useChatAllowed as jest.Mock;

const channelWithHostedStream = () => {
  return {
    hosting: {
      id: validId(randomId()),
      login: internet.userName(),
      stream: {
        id: validId(randomId()),
        type: StreamType.Live,
      },
    },
    stream: null,
  };
};

const channelWithNoStream = () => {
  return {
    stream: null,
  };
};

const mockProps = {
  channel: {
    broadcastSettings: {
      isMature: false,
    },
    displayName: internet.userName(),
    hosting: null,
    id: datatype.uuid(),
    login: internet.userName(),
    stream: {
      id: validId(randomId()),
      type: StreamType.Live,
      viewersCount: datatype.number(),
    },
  },
  currentUser: null,
  queryVariables: {
    login: internet.userName(),
    platform: Platform.StarshotDev,
    playerType: 'pulsar',
    skipPlayToken: true,
  },
};

describe(ChannelInterstitial, () => {
  const setup = createShallowWrapperFactory(
    ChannelInterstitial,
    () => ({} as any),
  );

  beforeEach(() => {
    mockUseStaticEnvironment.mockReturnValue({
      common: mockCommonDevelopmentEnvironment(Platform.Switch),
    });
    mockUseChatAllowed.mockReturnValue({
      isChatAllowed: () => true,
    });
  });

  it('redirects to ChannelHome if platform is not Switch', () => {
    mockUseStaticEnvironment.mockReturnValue({
      common: { platform: Platform.LG },
    });

    const { wrapper } = setup(mockProps);
    expect(wrapper.find(Redirect)).toExist();
  });

  it('redirects if the channel is hosting another channel', () => {
    const { wrapper } = setup({
      ...mockProps,
      channel: { ...mockProps.channel, ...channelWithHostedStream() },
    });
    expect(wrapper.find(Redirect)).toExist();
  });

  it('redirects if the channel is not live', () => {
    const { wrapper } = setup({
      ...mockProps,
      channel: { ...mockProps.channel, ...channelWithNoStream() },
    });
    expect(wrapper.find(Redirect)).toExist();
  });

  it('renders the not found UX when channel is not found', () => {
    const { wrapper } = setup({ channel: null });
    expect(wrapper.find(NotFoundError)).toExist();
  });

  it('does not render the not found UX when channel is found', () => {
    const { wrapper } = setup(mockProps);
    expect(wrapper.find(NotFoundError)).not.toExist();
  });

  it('does not render chat when it is not allowed', () => {
    mockUseChatAllowed.mockReturnValue({
      isChatAllowed: false,
    });
    const { wrapper } = setup(mockProps);
    expect(wrapper.find(InterstitialLayout).prop('chat')).toBeNull();
  });

  it('renders chat when it is allowed', () => {
    const { wrapper } = setup(mockProps);
    expect(wrapper.find(InterstitialLayout).prop('chat')).toBeTruthy();
  });
});
