import { useClosedCaptions } from 'pulsar';
import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import { useIntl } from 'tachyon-intl';
import { HorizontalNav, VerticalNav } from 'tachyon-tv-nav';
import {
  AlignItems,
  Display,
  FlexDirection,
  JustifyContent,
  Layout,
} from 'twitch-core-ui';
import { PLAYER_INITIAL_OFFSET_QUERY_PARAM_KEY } from '../../../../../config';
import { useFollow } from '../../../../../hooks';
import { RouteName, renderTachyonLink } from '../../../../../routing';
import {
  FocusableFollowButton,
  FocusableTextButton,
  PlaybackInfo,
  PlaybackOverlayItem,
} from '../../../../common';
import { ClosedCaptionsButton } from '../../../../player';
import { ClipSettingsButton } from './ClipSettingsButton';
import type { ClipBottomControls_clip } from './__generated__/ClipBottomControls_clip.graphql';

export type BottomControlsProps = {
  clip: ClipBottomControls_clip;
  focusIndex: number;
};

export const BottomControlsBase: FC<BottomControlsProps> = ({
  clip,
  focusIndex,
}) => {
  const { following } = useFollow(clip.broadcaster);
  const {
    available: captionsAvailable,
    enabled: captionsEnabled,
    toggle: toggleCaptions,
  } = useClosedCaptions();
  const { formatMessage } = useIntl();

  // Order matters for each of the buttons below
  const buttons = clip.broadcaster
    ? [
        <PlaybackOverlayItem key="settings">
          <ClipSettingsButton focusIndex={0} login={clip.broadcaster.login} />
        </PlaybackOverlayItem>,
      ]
    : [];

  // Only show the follow button if the channel isn't already followed to
  // avoid accidental unfollows.
  // This UX will be improved as part of: https://jira.xarth.tv/browse/EMP-3714
  if (!following && clip.broadcaster) {
    buttons.push(
      <PlaybackOverlayItem key="follow">
        <FocusableFollowButton
          channel={clip.broadcaster}
          focusIndex={buttons.length}
          iconOnly
          route={RouteName.Channel}
        />
      </PlaybackOverlayItem>,
    );
  }

  if (captionsAvailable) {
    buttons.push(
      <PlaybackOverlayItem key="captions">
        <ClosedCaptionsButton
          captionsEnabled={captionsEnabled}
          focusIndex={buttons.length}
          onClick={toggleCaptions}
        />
      </PlaybackOverlayItem>,
    );
  }

  const videoId = clip.video?.id;
  const verticalElementsCount = videoId ? 2 : 1;

  return (
    <HorizontalNav elementCount={2} focusIndex={focusIndex}>
      <Layout
        display={Display.Flex}
        fullWidth
        justifyContent={JustifyContent.Between}
      >
        {clip.broadcaster && (
          <PlaybackInfo
            channel={clip.broadcaster}
            focusIndex={0}
            gameDisplayName={clip.game?.displayName}
            isLive={false}
            recordedAt={clip.createdAt}
            title={clip.title}
            viewCount={clip.viewCount}
          />
        )}
        <VerticalNav elementCount={verticalElementsCount} focusIndex={1}>
          <Layout
            display={Display.Flex}
            flexDirection={FlexDirection.Column}
            justifyContent={JustifyContent.Center}
          >
            {videoId && (
              <Layout
                display={Display.Flex}
                flexShrink={0}
                fullWidth
                justifyContent={JustifyContent.End}
                margin={{ bottom: 1 }}
              >
                <PlaybackOverlayItem>
                  <FocusableTextButton
                    focusIndex={0}
                    linkTo="/deferToRenderLink"
                    renderLink={renderTachyonLink({
                      query: clip.videoOffsetSeconds
                        ? {
                            [PLAYER_INITIAL_OFFSET_QUERY_PARAM_KEY]: `${clip.videoOffsetSeconds}`,
                          }
                        : undefined,
                      route: RouteName.Vod,
                      routeParams: { id: videoId },
                    })}
                    withUnfocusBackground
                  >
                    {formatMessage('Watch Full Video', 'ClipBottomControls')}
                  </FocusableTextButton>
                </PlaybackOverlayItem>
              </Layout>
            )}
            <HorizontalNav elementCount={buttons.length} focusIndex={1}>
              <Layout
                alignItems={AlignItems.Center}
                display={Display.Flex}
                flexShrink={0}
                justifyContent={JustifyContent.End}
              >
                {buttons}
              </Layout>
            </HorizontalNav>
          </Layout>
        </VerticalNav>
      </Layout>
    </HorizontalNav>
  );
};

BottomControlsBase.displayName = 'BottomControlsBase';

export const ClipBottomControls = createFragmentContainer(BottomControlsBase, {
  clip: graphql`
    fragment ClipBottomControls_clip on Clip {
      broadcaster {
        ...useFollow_targetChannel @relay(mask: false)
        ...FocusableFollowButton_channel
        login
        ...PlaybackInfo_channel
      }
      createdAt
      viewCount
      video {
        id
      }
      videoOffsetSeconds
      title
      game {
        displayName
      }
    }
  `,
});

ClipBottomControls.displayName = 'ClipBottomControls';
