import { datatype, lorem } from 'faker';
import { mockClosedCaptionsContext } from 'pulsar';
import { createMountWrapperFactory } from 'tachyon-test-utils';
import { HorizontalNav, VerticalNav } from 'tachyon-tv-nav';
import { PLAYER_INITIAL_OFFSET_QUERY_PARAM_KEY } from '../../../../../config';
import { useFollow } from '../../../../../hooks';
import { RouteName, renderTachyonLink } from '../../../../../routing';
import {
  FocusableFollowButton,
  FocusableTextButton,
  PlaybackInfo,
} from '../../../../common';
import { ClosedCaptionsButton } from '../../../../player';
import { ClipSettingsButton } from './ClipSettingsButton';
import { BottomControlsBase } from '.';

jest.mock('../../../../../hooks');
const mockFollow = useFollow as jest.Mock;

jest.mock('../../../../common', () => ({
  ...jest.requireActual('../../../../common'),
  FocusableFollowButton: () => <div />,
  PlaybackInfo: () => <div />,
}));

jest.mock('../../../../../routing', () => ({
  ...jest.requireActual('../../../../../routing'),
  renderTachyonLink: jest.fn(() => () => <div />),
}));
const mockRenderTachyonLink = renderTachyonLink as jest.Mock;

describe(BottomControlsBase, () => {
  const setup = createMountWrapperFactory(
    BottomControlsBase,
    () => ({
      clip: {
        ' $refType': 'ClipBottomControls_clip',
        broadcaster: {
          ' $fragmentRefs': {
            FocusableFollowButton_channel: true,
            PlaybackInfo_channel: true,
          },
          displayName: 'therealderekt',
          id: datatype.uuid(),
          login: 'therealderekt',
          roles: {
            isPartner: true,
          },
          self: {
            follower: {
              followedAt: null,
            },
            subscriptionBenefit: null,
          },
        },
        createdAt: '2021-03-30T11:39:13Z',
        game: {
          displayName: lorem.words(2),
          name: lorem.words(2),
        },
        id: datatype.uuid(),
        slug: lorem.word(),
        thumbnailURL: lorem.word(),
        title: lorem.words(3),
        video: {
          id: datatype.uuid(),
        },
        videoOffsetSeconds: datatype.number(),
        viewCount: datatype.number(),
      },
      currentUser: {
        id: datatype.uuid(),
      },
      focusIndex: 0,
      visible: true,
    }),
    {
      wrappingContexts: () => [
        mockClosedCaptionsContext({
          available: false,
          enabled: false,
          subscribe: jest.fn(),
          toggle: jest.fn(),
        }),
      ],
    },
  );

  beforeEach(() => {
    mockFollow.mockReset();
    mockFollow.mockReturnValue({ following: false });
  });

  it('does not render the follow button if the user is already following', () => {
    mockFollow.mockReturnValue({ following: true });
    const { wrapper } = setup();
    expect(wrapper.find(FocusableFollowButton)).not.toExist();
  });

  it('does not render the captions button when unavailable', () => {
    const { wrapper } = setup();
    expect(wrapper.find(ClosedCaptionsButton)).not.toExist();
  });

  it('does not render the channel settings button if the broadcaster is null', () => {
    const { wrapper } = setup({ clip: { broadcaster: null } });
    expect(wrapper.find(ClipSettingsButton)).not.toExist();
  });

  it('assigns expected focusIndex values when all interactables are rendered', () => {
    mockFollow.mockReturnValue({ following: false });
    const { wrapper } = setup(undefined, {
      wrappingContexts: [
        mockClosedCaptionsContext({
          available: true,
        }),
      ],
    });

    // Outer horizontal nav should have the playback info + layout with the right buttons
    expect(wrapper.find(HorizontalNav).first()).toHaveProp({ elementCount: 2 });
    // We expect a vertical nav containing two elements (watch full video + the bottom row of buttons)
    expect(wrapper.find(VerticalNav)).toHaveProp({ elementCount: 2 });
    // Inner horizontal nav should contain the 3 buttons
    expect(wrapper.find(HorizontalNav).last()).toHaveProp({ elementCount: 3 });
    expect(wrapper.find(PlaybackInfo)).toHaveProp({ focusIndex: 0 });
    expect(wrapper.find(ClipSettingsButton)).toHaveProp({
      focusIndex: 0,
    });
    expect(wrapper.find(FocusableFollowButton)).toHaveProp({ focusIndex: 1 });
    expect(wrapper.find(ClosedCaptionsButton)).toHaveProp({ focusIndex: 2 });
  });

  it('does not render the watch full video button if the video is null', () => {
    const { wrapper } = setup({ clip: { video: null } });
    expect(wrapper.find(FocusableTextButton)).not.toExist();
    // We expect the vertical nav to only contain one element since the watch full video button is gone
    expect(wrapper.find(VerticalNav)).toHaveProp({ elementCount: 1 });
  });

  it('renders the watch full video button and routes to the vod page', () => {
    const { wrapper } = setup({
      clip: { video: { id: '12345' }, videoOffsetSeconds: 5555 },
    });
    expect(wrapper.find(FocusableTextButton)).toExist();

    expect(mockRenderTachyonLink).toHaveBeenCalledWith({
      query: { [PLAYER_INITIAL_OFFSET_QUERY_PARAM_KEY]: '5555' },
      route: RouteName.Vod,
      routeParams: {
        id: '12345',
      },
    });
  });
});
