import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import { VerticalNav } from 'tachyon-tv-nav';
import { Display, JustifyContent, Layout } from 'twitch-core-ui';
import { PlayingUILayout } from '../../../common';
import {
  ClipPlayer,
  HideablePlayPauseButton,
  MediaSession,
  PlayPauseButton,
  PlaybackScrubBar,
  ScrubControlsRoot,
} from '../../../player';
import type { ClipPlayerProps } from '../../../player';
import { ClipBottomControls } from './ClipBottomControls';
import type { ClipPlayingUI_clip } from './__generated__/ClipPlayingUI_clip.graphql';
import type { ClipPlayingUI_currentUser } from './__generated__/ClipPlayingUI_currentUser.graphql';

export type ClipPlayingUIProps = Pick<ClipPlayerProps, 'clipToken'> & {
  clip: ClipPlayingUI_clip;
  currentUser: ClipPlayingUI_currentUser | null;
  focusIndex: number;
};

// istanbul ignore next: trivial
export const ClipPlayingUIBase: FC<ClipPlayingUIProps> = ({
  clip,
  clipToken,
  currentUser,
  focusIndex,
}) => {
  let overlayChildren, playerControls;

  // If a clip is less than 30 seconds disable the progress bar and
  // opt into a simplistic control panel with just play/pause.
  // This is a workaround because 1 second is our current unit of precision
  // and 4 second clips look bad jumping across the screen at this rate.
  // There's also no point in scrubbing on clips shorter than 30 seconds.
  if (clip.durationSeconds >= 30) {
    overlayChildren = <HideablePlayPauseButton />;
    playerControls = (
      <Layout padding={{ bottom: 1 }}>
        <MediaSession />
        <PlaybackScrubBar focusIndex={1} />
      </Layout>
    );
  } else {
    playerControls = (
      <Layout
        display={Display.Flex}
        justifyContent={JustifyContent.Around}
        padding={{ bottom: 1 }}
      >
        <MediaSession />
        <PlayPauseButton focusIndex={1} />
      </Layout>
    );
  }

  return (
    <VerticalNav elementCount={3} focusIndex={focusIndex}>
      <PlayingUILayout
        channel={clip.broadcaster}
        live={false}
        overlayChildren={overlayChildren}
        player={
          <ClipPlayer
            autoPlay
            clip={clip}
            clipToken={clipToken}
            currentUser={currentUser}
            loop
          />
        }
      >
        <PlayingUILayout.BackButton focusIndex={0} />
        <PlayingUILayout.Controls>
          {playerControls}
          <ClipBottomControls clip={clip} focusIndex={2} />
        </PlayingUILayout.Controls>
      </PlayingUILayout>
    </VerticalNav>
  );
};

ClipPlayingUIBase.displayName = 'ClipPlayingUIBase';

export const ClipPlayingUI = createFragmentContainer(
  (props: ClipPlayingUIProps) => (
    <ScrubControlsRoot scrubIntervalSeconds={1}>
      <ClipPlayingUIBase {...props} />
    </ScrubControlsRoot>
  ),
  {
    clip: graphql`
      fragment ClipPlayingUI_clip on Clip {
        ...ClipPlayer_clip
        ...ClipBottomControls_clip
        durationSeconds
        broadcaster {
          displayName
          login
          roles {
            isAffiliate
            isPartner
          }
        }
      }
    `,
    currentUser: graphql`
      fragment ClipPlayingUI_currentUser on User {
        ...ClipPlayer_currentUser
      }
    `,
  },
);
