import {
  ClipPlaybackRestrictionsRoot,
  PlayerControllerRoot,
  PlayerType,
} from 'pulsar';
import { graphql } from 'react-relay/hooks';
import { Platform, useStaticEnvironment } from 'tachyon-environment';
import type {
  ClipPathParameters,
  ClipInitialProps as DefaultClipInitialProps,
} from 'tachyon-page-utils';
import {
  clipGetInitialProps,
  clipIsFound,
  clipIsNotFoundServerside,
  clipPageviewTracking,
  getCurrentUser,
} from 'tachyon-page-utils';
import { isBrowser } from 'tachyon-utils';
import { Redirect, RouteName } from '../../../routing';
import { StarshotMain } from '../../common';
import { NotFoundError } from '../../errors';
import { ClipRestriction } from '../../player';
import type { StarshotPage } from '../types';
import { ClipPlayingUI } from './ClipPlayingUI';
import type {
  Clip_QueryResponse,
  Clip_QueryVariables,
} from './__generated__/Clip_Query.graphql';

export type ClipInitialProps = DefaultClipInitialProps<Clip_QueryVariables>;

export type ClipProps = Clip_QueryResponse & ClipInitialProps;

export const Clip: StarshotPage<
  ClipInitialProps,
  ClipProps,
  ClipPathParameters
> = ({ clip, currentUser, login, queryVariables, ...clipToken }) => {
  const {
    common: { platform },
  } = useStaticEnvironment();

  if (!clipIsFound(clip)) {
    return <NotFoundError />;
  }

  if (platform === Platform.Switch) {
    return (
      <Redirect
        params={{
          route: RouteName.ClipInterstitial,
          routeParams: {
            login: clip.broadcaster?.login ?? 'unknown',
            slug: clip.slug,
          },
        }}
        replace
      />
    );
  }

  return (
    <StarshotMain>
      <PlayerControllerRoot>
        <ClipPlaybackRestrictionsRoot clip={clip}>
          <ClipRestriction clip={clip} focusIndex={0}>
            <ClipPlayingUI
              clip={clip}
              clipToken={clipToken}
              currentUser={currentUser}
              focusIndex={0}
            />
          </ClipRestriction>
        </ClipPlaybackRestrictionsRoot>
      </PlayerControllerRoot>
    </StarshotMain>
  );
};

Clip.currentUser = getCurrentUser;
Clip.displayName = 'Clip';
Clip.isNotFoundServerside = clipIsNotFoundServerside;
Clip.navigationBehavior = () => ({});
Clip.pageviewTracking = clipPageviewTracking;
Clip.getInitialProps = (context) =>
  clipGetInitialProps(context, {
    platform: context.platform,
    playerType: PlayerType.Pulsar,
    skipPlayToken: !isBrowser(),
  });

Clip.query = graphql`
  query Clip_Query(
    $slug: ID!
    $platform: String!
    $playerType: String!
    $skipPlayToken: Boolean!
  ) {
    clip(slug: $slug) {
      ...ClipPlayingUI_clip
      ...ClipRestriction_clip @relay(mask: false)
      id
      broadcaster {
        id
        login
      }
      game {
        name
      }
      slug
    }
    currentUser {
      ...types_currentUser @relay(mask: false)
      ...ClipPlayingUI_currentUser
    }
    ...ClipPlayer_token
  }
`;
