import {
  ClipPlaybackRestrictionsRoot,
  PlayerControllerRoot,
  PlayerType,
} from 'pulsar';
import { graphql } from 'react-relay/hooks';
import { Platform, useStaticEnvironment } from 'tachyon-environment';
import { StreamType, getStreamType } from 'tachyon-more-ui';
import type {
  ClipPathParameters,
  ClipInitialProps as DefaultClipInitialProps,
} from 'tachyon-page-utils';
import {
  clipGetInitialProps,
  clipIsFound,
  clipIsNotFoundServerside,
  clipPageviewTracking,
  getCurrentUser,
} from 'tachyon-page-utils';
import { isBrowser } from 'tachyon-utils';
import {
  useInterstitialPlayAttempt,
  useIsClipPlayableOnSwitch,
} from '../../../hooks';
import { Redirect, RouteName } from '../../../routing';
import {
  InterstitialLayout,
  InterstitialVideoDetails,
  StarshotMain,
} from '../../common';
import { NotFoundError } from '../../errors';
import { ClipPlayer, ClipRestriction } from '../../player';
import type { Clip_QueryVariables } from '../Clip/__generated__/Clip_Query.graphql';
import type { StarshotPage } from '../types';
import type { ClipInterstitial_QueryResponse } from './__generated__/ClipInterstitial_Query.graphql';

export type ClipInitialProps = DefaultClipInitialProps<Clip_QueryVariables>;

export type ClipInterstitialProps = ClipInitialProps &
  ClipInterstitial_QueryResponse;

export const ClipInterstitial: StarshotPage<
  ClipInitialProps,
  ClipInterstitialProps,
  ClipPathParameters
> = ({ clip, currentUser, login, queryVariables, ...clipToken }) => {
  const { attemptedPlay, resetAttemptedPlay, triggerAttemptedPlay } =
    useInterstitialPlayAttempt();
  const {
    common: { platform },
  } = useStaticEnvironment();
  const { isContentPlayable, setManifestUrl } = useIsClipPlayableOnSwitch();

  if (!clipIsFound(clip)) {
    return <NotFoundError />;
  }

  if (platform !== Platform.Switch) {
    return (
      <Redirect
        params={{
          route: RouteName.Clip,
          routeParams: {
            login: queryVariables.login ?? 'unknown',
            slug: queryVariables.slug,
          },
        }}
        replace
      />
    );
  }

  const streamType = getStreamType({
    broadcaster: clip.broadcaster,
    type: clip.broadcaster?.stream?.type ?? null,
  });

  return (
    <StarshotMain>
      <PlayerControllerRoot>
        <ClipPlaybackRestrictionsRoot clip={clip}>
          <ClipRestriction
            clip={clip}
            focusIndex={0}
            passThrough={!attemptedPlay}
          >
            <InterstitialLayout
              attemptedPlay={attemptedPlay}
              channel={clip.broadcaster}
              contentType="clip"
              fullVideoID={clip.video?.id}
              isChannelLive={streamType === StreamType.Live}
              isContentPlayable={isContentPlayable}
              player={
                <ClipPlayer
                  autoPlay={false}
                  clip={clip}
                  clipToken={clipToken}
                  currentUser={currentUser}
                  onManifestUrlCreated={setManifestUrl}
                />
              }
              resetAttemptedPlay={resetAttemptedPlay}
              triggerPlayAttempt={triggerAttemptedPlay}
            >
              <InterstitialVideoDetails
                categoryImageURL={clip.game?.boxArtURL}
                categoryName={clip.game?.displayName}
                createdAt={clip.createdAt}
                videoDuration={clip.durationSeconds}
                videoTitle={clip.title}
              />
            </InterstitialLayout>
          </ClipRestriction>
        </ClipPlaybackRestrictionsRoot>
      </PlayerControllerRoot>
    </StarshotMain>
  );
};

ClipInterstitial.currentUser = getCurrentUser;
ClipInterstitial.displayName = 'ClipInterstitial';
ClipInterstitial.isNotFoundServerside = clipIsNotFoundServerside;
ClipInterstitial.navigationBehavior = () => ({ displayNavMenu: true });
ClipInterstitial.pageviewTracking = clipPageviewTracking;
ClipInterstitial.getInitialProps = (context) =>
  clipGetInitialProps(context, {
    platform: context.platform,
    playerType: PlayerType.Pulsar,
    skipPlayToken: !isBrowser(),
  });

ClipInterstitial.query = graphql`
  query ClipInterstitial_Query(
    $slug: ID!
    $platform: String!
    $playerType: String!
    $skipPlayToken: Boolean!
  ) {
    clip(slug: $slug) {
      ...ClipPlayer_clip
      ...ClipRestriction_clip @relay(mask: false)
      id
      broadcaster {
        ...InterstitialLayout_channel
        id
        login
        hosting {
          id
        }
        stream {
          id
          type
        }
      }
      createdAt
      durationSeconds
      game {
        boxArtURL
        displayName
        name
      }
      slug
      title
      video {
        id
      }
    }
    currentUser {
      ...types_currentUser @relay(mask: false)
      ...ClipPlayer_currentUser
    }
    ...ClipPlayer_token
  }
`;
