import type { FC } from 'react';
import { useCallback, useMemo } from 'react';
import { VerticalNav } from 'tachyon-tv-nav';
import type { OmitRefType } from 'tachyon-type-library';
import { Layout } from 'twitch-core-ui';
import {
  GRID_CARD_BOTTOM_MARGIN_REM,
  OVERSCAN_PADDING_REM,
  WATCHABLE_CARD_CONFIG,
} from '../../../../config';
import {
  BannerPage,
  FocusableStreamCard,
  FocusableTower,
  useBannerData,
} from '../../../common';
import type { GameDirectory_stream } from '../__generated__/GameDirectory_stream.graphql';
import type { GameDirectoryGame } from './GameDirectoryBannerContent';
import { GameDirectoryBannerContent } from './GameDirectoryBannerContent';

export type { GameDirectoryGame } from './GameDirectoryBannerContent';

export type GameDirectoryStreamList = Array<OmitRefType<GameDirectory_stream>>;

export const FULL_ONSCREEN_ROWS = 1;

export type GameDirectoryContentProps = {
  focusIndex: number;
  game: GameDirectoryGame;
  streams: GameDirectoryStreamList;
};

export const GameDirectoryContent: FC<GameDirectoryContentProps> = ({
  focusIndex,
  game,
  streams,
}) => {
  const updateBannerData = useBannerData({
    backgroundImageSrc: streams[0].previewImageURL,
    categoryImageSrc: game.boxArtURL,
  });

  const onCardFocus = useCallback(
    (index: number) => {
      updateBannerData({
        backgroundImageSrc: streams[index].previewImageURL,
        categoryImageSrc: game.boxArtURL,
      });
    },
    [game, streams, updateBannerData],
  );

  const streamItemRenderer = useCallback(
    (stream: OmitRefType<GameDirectory_stream>, streamFocusIndex: number) => (
      <FocusableStreamCard
        focusIndex={streamFocusIndex}
        key={stream.id}
        marginBottom={GRID_CARD_BOTTOM_MARGIN_REM}
        onFocus={onCardFocus}
        stream={stream}
      />
    ),
    [onCardFocus],
  );

  const bannerContentProps = useMemo(
    () => ({
      game,
    }),
    [game],
  );

  return (
    <VerticalNav
      elementCount={2}
      focusIndex={focusIndex}
      initialChildFocusIndex={1}
    >
      <BannerPage
        BannerContent={GameDirectoryBannerContent}
        bannerContentProps={bannerContentProps}
        showCategoryImage
        updateBannerData={updateBannerData}
      >
        <Layout padding={{ top: 1, x: OVERSCAN_PADDING_REM }}>
          <FocusableTower
            config={WATCHABLE_CARD_CONFIG}
            focusIndex={1}
            fullOnScreenCardRows={FULL_ONSCREEN_ROWS}
            itemRenderer={streamItemRenderer}
            items={streams}
          />
        </Layout>
      </BannerPage>
    </VerticalNav>
  );
};
GameDirectoryContent.displayName = 'GameDirectoryContent';
