import { graphql } from 'react-relay/hooks';
import { useIntl } from 'tachyon-intl';
import type { GamePathParameters } from 'tachyon-page-utils';
import {
  gameGetInitialProps,
  gamePageviewTracking,
  getCurrentUser,
} from 'tachyon-page-utils';
import { NodeNav } from 'tachyon-tv-nav';
import { reduceToNonNullNodes } from 'tachyon-utils';
import {
  AlignItems,
  CoreText,
  Display,
  FlexDirection,
  FontSize,
  JustifyContent,
  Layout,
} from 'twitch-core-ui';
import { useTachyonRouter } from '../../../routing';
import { FocusableTextButton, StarshotMain } from '../../common';
import type { StarshotPage } from '../types';
import { GameDirectoryContent } from './GameDirectoryContent';
import type {
  GameDirectory_QueryResponse,
  GameDirectory_QueryVariables,
} from './__generated__/GameDirectory_Query.graphql';

interface GameDirectoryInitialProps {
  queryVariables: GameDirectory_QueryVariables;
}

export interface GameDirectoryProps
  extends GameDirectoryInitialProps,
    GameDirectory_QueryResponse {}

export const REQUESTED_STREAM_COUNT = 40;

export const GameDirectory: StarshotPage<
  GameDirectoryInitialProps,
  GameDirectoryProps,
  GamePathParameters
> = ({ game, queryVariables }) => {
  const { formatMessage } = useIntl();
  const streams = reduceToNonNullNodes(game?.streams?.edges);
  const { back } = useTachyonRouter();

  let content: JSX.Element;
  if (!game || streams.length === 0) {
    content = (
      <NodeNav focusIndex={0} takeFocusOnFirstRender>
        <Layout
          alignItems={AlignItems.Center}
          display={Display.Flex}
          flexDirection={FlexDirection.Column}
          fullHeight
          justifyContent={JustifyContent.Center}
          padding={{ bottom: 5 }}
        >
          <CoreText fontSize={FontSize.Size2}>
            {formatMessage(
              'There are no live channels for {fallbackGameTitle}.',
              { fallbackGameTitle: game?.name ?? queryVariables.gameAlias },
              'Game404',
            )}
          </CoreText>
          <Layout padding={{ top: 1 }}>
            <FocusableTextButton focusIndex={0} onClick={back}>
              {formatMessage('Go back', 'GameDirectoryEmpty')}
            </FocusableTextButton>
          </Layout>
        </Layout>
      </NodeNav>
    );
  } else {
    content = (
      <GameDirectoryContent focusIndex={0} game={game} streams={streams} />
    );
  }

  return <StarshotMain>{content}</StarshotMain>;
};

GameDirectory.currentUser = getCurrentUser;
GameDirectory.displayName = 'GameDirectory';
GameDirectory.navigationBehavior = () => ({
  displayBackButton: true,
  displayNavMenu: true,
});
GameDirectory.pageviewTracking = gamePageviewTracking;
GameDirectory.getInitialProps = (context) =>
  gameGetInitialProps(context, {
    channelsCount: REQUESTED_STREAM_COUNT,
  });

GameDirectory.query = graphql`
  query GameDirectory_Query($gameAlias: String!, $channelsCount: Int!) {
    currentUser {
      ...types_currentUser @relay(mask: false)
    }
    game(name: $gameAlias) {
      ...GameDirectory_game @relay(mask: false)
    }
  }
`;

// eslint-disable-next-line no-unused-expressions
graphql`
  fragment GameDirectory_stream on Stream {
    id
    previewImageURL
    ...FocusableStreamCard_stream
  }
`;

// eslint-disable-next-line no-unused-expressions
graphql`
  fragment GameDirectory_game on Game {
    boxArtURL
    displayName
    name
    streams(first: $channelsCount) {
      edges {
        node {
          ...GameDirectory_stream @relay(mask: false)
        }
      }
    }
  }
`;
