import type { ReactElement } from 'react';
import { graphql } from 'react-relay/hooks';
import styled from 'styled-components';
import { defaultPageviewTracking, getCurrentUser } from 'tachyon-page-utils';
import type { DeepNonNullableObj } from 'tachyon-type-library';
import { reduceToNonNullNodes } from 'tachyon-utils';
import {
  CATEGORY_CARD_CONFIG,
  GRID_CARD_BOTTOM_MARGIN_REM,
  OVERSCAN_PADDING_REM,
  TOP_NAV_HEIGHT_REM,
} from '../../../config';
import {
  FocusableCategoryCard,
  FocusableTower,
  StarshotMain,
} from '../../common';
import { InternalServerError } from '../../errors';
import type { StarshotPage } from '../types';
import type {
  GamesDirectory_QueryResponse,
  GamesDirectory_QueryVariables,
} from './__generated__/GamesDirectory_Query.graphql';

export const REQUESTED_GAME_COUNT = 80;

const ScGamesDirectoryContainer = styled.div`
  height: 100%;
  margin-top: ${TOP_NAV_HEIGHT_REM}rem;
  padding: 3rem ${OVERSCAN_PADDING_REM}rem 0;
  position: fixed;
`;

type CategoryNode =
  DeepNonNullableObj<GamesDirectory_QueryResponse>['games']['edges'][0]['node'];

function categoryItemRenderer(
  category: CategoryNode,
  focusIndex: number,
): ReactElement {
  return (
    <FocusableCategoryCard
      category={category}
      focusIndex={focusIndex}
      key={focusIndex}
      marginBottom={GRID_CARD_BOTTOM_MARGIN_REM}
    />
  );
}

interface GamesDirectoryInitialProps {
  queryVariables: GamesDirectory_QueryVariables;
}

export interface GamesDirectoryProps
  extends GamesDirectoryInitialProps,
    GamesDirectory_QueryResponse {}

/**
 * This value must be validated against card and tower placeholder width to ensure
 * vertical navigation works as expected.
 */
const FULL_ONSCREEN_ROWS = 2;

export const GamesDirectory: StarshotPage<
  GamesDirectoryInitialProps,
  GamesDirectoryProps
> = ({ games }) => {
  const gameNodes = reduceToNonNullNodes(games?.edges);

  // This state is reachable due to the presence of the retry button
  // we will probably want a more elegant solution down the road
  // https://jira.twitch.com/browse/EMP-3271
  if (gameNodes.length === 0) {
    return <InternalServerError />;
  }

  // TopNav gets a background here to hide the upper game cards as we scroll
  // downward
  return (
    <StarshotMain>
      <ScGamesDirectoryContainer>
        <FocusableTower
          config={CATEGORY_CARD_CONFIG}
          focusIndex={0}
          fullOnScreenCardRows={FULL_ONSCREEN_ROWS}
          itemRenderer={categoryItemRenderer}
          items={gameNodes}
        />
      </ScGamesDirectoryContainer>
    </StarshotMain>
  );
};

GamesDirectory.currentUser = getCurrentUser;
GamesDirectory.displayName = 'GamesDirectory';
GamesDirectory.navigationBehavior = () => ({
  displayNavMenu: true,
  topNavBackground: true,
});
GamesDirectory.pageviewTracking = defaultPageviewTracking;

GamesDirectory.getInitialProps = () => ({
  queryVariables: {
    first: REQUESTED_GAME_COUNT,
  },
});

GamesDirectory.query = graphql`
  query GamesDirectory_Query($first: Int!) {
    currentUser {
      ...types_currentUser @relay(mask: false)
    }
    games(first: $first) {
      edges {
        node {
          ...FocusableCategoryCard_category
        }
      }
    }
  }
`;
