import { createShallowWrapperFactory } from 'tachyon-test-utils';
import { FocusableTower } from '../../common';
import { InternalServerError } from '../../errors';
import type { StarshotPageContext } from '../types';
import { GamesDirectory, REQUESTED_GAME_COUNT } from '.';

describe(GamesDirectory, () => {
  describe(GamesDirectory.getInitialProps!, () => {
    it('gets data to display the first page of games', async () => {
      const {
        queryVariables: { first },
      } = await GamesDirectory.getInitialProps!({} as StarshotPageContext);
      expect(first).toEqual(REQUESTED_GAME_COUNT);
    });
  });

  describe('rendering', () => {
    const setup = createShallowWrapperFactory(GamesDirectory, () => ({
      currentUser: null,
      games: {
        edges: [
          {
            node: {
              ' $fragmentRefs': { FocusableCategoryCard_category: true },
            },
          },
          {
            node: {
              ' $fragmentRefs': { FocusableCategoryCard_category: true },
            },
          },
          {
            node: {
              ' $fragmentRefs': { FocusableCategoryCard_category: true },
            },
          },
        ],
      },
      queryVariables: { first: REQUESTED_GAME_COUNT },
    }));

    it('renders internal error when games are null', () => {
      const { wrapper } = setup({ games: null });

      expect(wrapper.find(InternalServerError)).toExist();
    });

    it('renders internal error when games edges are null', () => {
      const { wrapper } = setup({ games: { edges: null } });

      expect(wrapper.find(InternalServerError)).toExist();
    });

    it('renders internal error when games edges are empty', () => {
      const { wrapper } = setup({ games: { edges: [] } });

      expect(wrapper.find(InternalServerError)).toExist();
    });

    it('renders internal error when games edges have only null nodes', () => {
      const { wrapper } = setup({ games: { edges: [{ node: null }] } });
      expect(wrapper.find(InternalServerError)).toExist();
    });

    it('does not render internal error when valid fragment ref is provided', () => {
      const { wrapper } = setup();
      expect(wrapper.find(InternalServerError)).not.toExist();
    });

    it('renders a tower when game data is populated', () => {
      const { props, wrapper } = setup();
      expect(wrapper.find(FocusableTower).prop('items')).toHaveLength(
        props.games!.edges!.length,
      );
    });
  });
});
