import { useCallback, useMemo } from 'react';
import { graphql } from 'react-relay/hooks';
import { useDiscoveryTracking } from 'tachyon-discovery';
import { useIntl } from 'tachyon-intl';
import { defaultPageviewTracking, getCurrentUser } from 'tachyon-page-utils';
import { uniqueIDGenerator, useCallbackMemoizedByKey } from 'tachyon-utils';
import { CATEGORY_CARD_CONFIG, WATCHABLE_CARD_CONFIG } from '../../../config';
import { RouteName, renderTachyonLink } from '../../../routing';
import {
  BannerPage,
  HorizontalShelf,
  StarshotMain,
  VerticalShelfList,
  useBannerData,
} from '../../common';
import { InternalServerError } from '../../errors';
import type { StarshotPage } from '../types';
import { HomepageBannerContent } from './HomepageBannerContent';
import { RecommendationItemRenderer } from './RecommendationItemRenderer';
import type {
  Homepage_QueryResponse,
  Homepage_QueryVariables,
} from './__generated__/Homepage_Query.graphql';
import { shelfTitle } from './shelfTitle';
import type { Stream } from './utils';
import { extractStream, getRecommendationShelves, shelfType } from './utils';

export const ID_LENGTH = 16;
export const RECOMMENDATION_ITEMS_PER_ROW = 20;
export const RECOMMENDATION_NUM_OF_SHELVES = 8;

type HomepageInitialProps = {
  queryVariables: Homepage_QueryVariables;
};

export type HomepageProps = Homepage_QueryResponse & HomepageInitialProps;

export const Homepage: StarshotPage<HomepageInitialProps, HomepageProps> = ({
  shelves,
}) => {
  const { initialBannerStream, renderableShelves } = useMemo(() => {
    const _shelves = getRecommendationShelves(shelves);

    return {
      initialBannerStream:
        _shelves.length === 0
          ? null
          : extractStream(_shelves[0].cards[0] ?? null),
      renderableShelves: _shelves,
    };
  }, [shelves]);
  const updateBannerData = useBannerData<Stream | null>({
    backgroundImageSrc: initialBannerStream?.previewImageURL,
    categoryImageSrc: initialBannerStream?.game?.boxArtURL,
    contentData: initialBannerStream,
  });

  // TODO: consider pushing the display memo logic into Discovery Tracking and reset
  // based on routeName change. This would allow RecommendationItemRenderer
  // to consume directly even with virtualization
  const { onRecommendationItemDisplay } = useDiscoveryTracking();
  const memoOnRecommendationItemDisplay = useCallbackMemoizedByKey(
    onRecommendationItemDisplay,
    {
      targetKeys: ['itemID', 'modelTrackingID'],
    },
  );

  const onCardFocus = useCallback(
    (activeCard: Stream | null) => {
      updateBannerData({
        backgroundImageSrc: activeCard?.previewImageURL ?? null,
        categoryImageSrc: activeCard?.game?.boxArtURL ?? null,
        contentData: activeCard,
      });
    },
    [updateBannerData],
  );

  const { formatMessage } = useIntl();

  // This state is reachable due to the presence of the retry button
  // we will probably want a more elegant solution down the road
  // https://jira.twitch.com/browse/EMP-3271
  if (renderableShelves.length === 0) {
    return (
      <InternalServerError
        secondaryAction={{
          children: formatMessage('Browse', 'Browse'),
          linkTo: '/deferToRenderLink',
          renderLink: renderTachyonLink({
            route: RouteName.GamesDirectory,
          }),
        }}
      />
    );
  }

  return (
    <StarshotMain>
      <BannerPage
        BannerContent={HomepageBannerContent}
        showCategoryImage
        updateBannerData={updateBannerData}
      >
        <VerticalShelfList
          focusIndex={0}
          items={renderableShelves}
          shelfRenderer={(shelf, shelfIndex) => {
            const config =
              shelfType(shelf) === 'Game'
                ? CATEGORY_CARD_CONFIG
                : WATCHABLE_CARD_CONFIG;

            return (
              <HorizontalShelf
                config={config}
                focusIndex={shelfIndex}
                itemRenderer={(node, cardIndex) => (
                  <RecommendationItemRenderer
                    cardIndex={cardIndex}
                    key={cardIndex}
                    node={node}
                    onDisplay={memoOnRecommendationItemDisplay}
                    onFocus={onCardFocus}
                  />
                )}
                items={shelf.cards}
                key={shelf.id}
                title={shelfTitle(shelf.title)}
                virtualized
              />
            );
          }}
          takeFocusOnFirstRender
        />
      </BannerPage>
    </StarshotMain>
  );
};

Homepage.currentUser = getCurrentUser;
Homepage.displayName = 'Homepage';
Homepage.navigationBehavior = () => ({
  displayNavMenu: true,
});
Homepage.pageviewTracking = defaultPageviewTracking;

// eslint-disable-next-line no-unused-expressions
graphql`
  fragment Homepage_title on ShelfTitle {
    fallbackLocalizedTitle
    localizedTitleTokens {
      node {
        ... on Game {
          __typename
          displayName
          name
        }
        ... on TextToken {
          __typename
          text
          location
        }
      }
    }
  }
`;

// eslint-disable-next-line no-unused-expressions
graphql`
  fragment Homepage_streamBanner on Stream {
    previewImageURL
    broadcaster {
      displayName
      broadcastSettings {
        title
      }
    }
    game {
      displayName
      boxArtURL
    }
  }
`;

// eslint-disable-next-line no-unused-expressions
graphql`
  fragment Homepage_stream on Stream {
    id
    previewImageURL
    ...Homepage_streamBanner @relay(mask: false)
  }
`;

// eslint-disable-next-line no-unused-expressions
graphql`
  fragment Homepage_card on ShelfContent {
    __typename
    ... on Stream {
      ...Homepage_stream @relay(mask: false)
      ...FocusableStreamCard_stream
    }
    ... on Game {
      ...FocusableCategoryCard_category
      id
      streams(first: 1) {
        edges {
          node {
            ...Homepage_stream @relay(mask: false)
          }
        }
      }
    }
  }
`;

// eslint-disable-next-line no-unused-expressions
graphql`
  fragment Homepage_tracking on Shelf {
    trackingInfo {
      reasonTarget
      reasonTargetType
      reasonType
      rowName
    }
    content {
      edges {
        trackingID
      }
    }
  }
`;

// eslint-disable-next-line no-unused-expressions
graphql`
  fragment Homepage_shelf on Shelf {
    id
    title {
      ...Homepage_title @relay(mask: false)
    }
  }
`;

// eslint-disable-next-line no-unused-expressions
graphql`
  fragment Homepage_shelves on ShelfConnection {
    edges {
      node {
        ...Homepage_shelf @relay(mask: false)
        ...Homepage_tracking @relay(mask: false)
        content {
          edges {
            node {
              ...Homepage_card @relay(mask: false)
            }
          }
        }
      }
    }
  }
`;

Homepage.getInitialProps = ({ platform }) => ({
  queryVariables: {
    itemsPerRow: RECOMMENDATION_ITEMS_PER_ROW,
    limit: RECOMMENDATION_NUM_OF_SHELVES,
    platform,
    requestID: uniqueIDGenerator(ID_LENGTH),
  },
});

Homepage.query = graphql`
  query Homepage_Query(
    $itemsPerRow: Int!
    $limit: Int!
    $platform: String!
    $requestID: String!
  ) {
    currentUser {
      ...types_currentUser @relay(mask: false)
    }
    shelves(
      itemsPerRow: $itemsPerRow
      first: $limit
      platform: $platform
      requestID: $requestID
    ) {
      ...Homepage_shelves @relay(mask: false)
    }
  }
`;
