import { internet, lorem } from 'faker';
import { Platform } from 'tachyon-environment';
import { validId } from 'tachyon-relay';
import { createMountWrapperFactory, randomId } from 'tachyon-test-utils';
import { uniqueIDGenerator } from 'tachyon-utils';
import {
  FocusableCategoryCard,
  FocusableStreamCard,
  HorizontalShelf,
  useBannerData,
} from '../../common';
import { InternalServerError } from '../../errors';
import { GAME_TOKEN } from './shelfTitle';
import { extractStream } from './utils';
import {
  Homepage,
  ID_LENGTH,
  RECOMMENDATION_ITEMS_PER_ROW,
  RECOMMENDATION_NUM_OF_SHELVES,
} from '.';

jest.mock('../../common', () => {
  const { MockHorizontalShelf } = jest.requireActual(
    '../../common/HorizontalShelf/test-mocks',
  );
  const { MockVerticalShelfList } = jest.requireActual(
    '../../common/VerticalShelfList/test-mocks',
  );
  return {
    ...jest.requireActual('../../common'),
    FocusableCategoryCard: jest.fn(() => <div />),
    FocusableStreamCard: jest.fn(() => <div />),
    HorizontalShelf: MockHorizontalShelf,
    VerticalShelfList: MockVerticalShelfList,
    useBannerData: jest.fn(),
  };
});
const mockFocusableCategoryCard = FocusableCategoryCard as jest.Mock;
const mockFocusableStreamCard = FocusableStreamCard as jest.Mock;
const mockUseBannerData = useBannerData as jest.Mock;

const mockStreamCardQuery = () =>
  ({
    ' $fragmentRefs': {
      FocusableStreamCard_stream: true,
    },
    __typename: 'Stream',
    broadcaster: {
      broadcastSettings: {
        title: lorem.word(),
      },
      displayName: lorem.word(),
    },
    game: {
      boxArtURL: internet.url(),
      displayName: lorem.word(),
    },
    id: validId(randomId()),
    previewImageURL: internet.url(),
  } as const);

const mockCategoryCardQuery = () =>
  ({
    ' $fragmentRefs': {
      FocusableCategoryCard_category: true,
    },
    __typename: 'Game',
    id: validId(randomId()),
    streams: { edges: [{ node: mockStreamCardQuery() }] },
  } as const);

const mockShelfQuery = () =>
  ({
    content: {
      edges: [
        {
          node: mockStreamCardQuery(),
          trackingID: randomId(),
        },
        {
          node: mockCategoryCardQuery(),
          trackingID: randomId(),
        },
      ],
    },
    id: validId(randomId()),
    title: {
      fallbackLocalizedTitle: 'bar',
      localizedTitleTokens: [],
    },
    trackingInfo: {
      reasonTarget: lorem.word(),
      reasonTargetType: lorem.word(),
      reasonType: lorem.word(),
      rowName: lorem.word(),
    },
  } as const);

const mockStreamShelfEdge = () =>
  ({
    node: {
      ...mockShelfQuery(),
      content: {
        edges: [
          { node: mockStreamCardQuery() },
          { node: mockStreamCardQuery() },
        ],
      },
    },
  } as const);

const mockGameShelfEdge = () =>
  ({
    node: {
      ...mockShelfQuery(),
      content: {
        edges: [
          { node: mockCategoryCardQuery() },
          { node: mockCategoryCardQuery() },
        ],
      },
    },
  } as const);

describe(Homepage, () => {
  const setup = createMountWrapperFactory(Homepage, () => ({
    currentUser: null,
    focusedCard: null,
    queryVariables: {
      itemsPerRow: RECOMMENDATION_ITEMS_PER_ROW,
      limit: RECOMMENDATION_NUM_OF_SHELVES,
      platform: Platform.StarshotDev,
      requestID: uniqueIDGenerator(ID_LENGTH),
    },
    shelves: null,
  }));

  describe('sets the initial background image', () => {
    it('when first shelf is streams', () => {
      const { props } = setup({
        shelves: {
          edges: [mockStreamShelfEdge()],
        },
      });

      const firstCardStream = extractStream(
        props.shelves!.edges![0].node.content.edges![0].node as any,
      )!;

      expect(mockUseBannerData).toHaveBeenCalledWith({
        backgroundImageSrc: firstCardStream.previewImageURL,
        categoryImageSrc: firstCardStream.game!.boxArtURL,
        // stream node enhanced with tracking
        contentData: expect.objectContaining(firstCardStream),
      });
    });

    it('when first shelf is games', () => {
      const { props } = setup({
        shelves: {
          edges: [mockGameShelfEdge()],
        },
      });

      const firstCardStream = extractStream(
        props.shelves!.edges![0].node.content.edges![0].node as any,
      )!;

      expect(mockUseBannerData).toHaveBeenCalledWith({
        backgroundImageSrc: firstCardStream.previewImageURL,
        categoryImageSrc: firstCardStream.game!.boxArtURL,
        // stream node enhanced with tracking
        contentData: expect.objectContaining(firstCardStream),
      });
    });
  });

  describe('shelf', () => {
    it('renders the title', () => {
      const { wrapper } = setup({
        shelves: {
          edges: [
            {
              node: {
                ...mockShelfQuery(),
                title: {
                  fallbackLocalizedTitle: 'bar',
                  localizedTitleTokens: [
                    {
                      node: {
                        __typename: GAME_TOKEN,
                        displayName: 'foo',
                        name: lorem.word(),
                      },
                    },
                  ],
                },
              },
            },
          ],
        },
      });

      expect(wrapper.find(HorizontalShelf)).toHaveProp({ title: 'Foo' });
    });
  });

  describe('cards', () => {
    it('renders a CategoryCard for each Game content node', () => {
      setup({
        shelves: {
          edges: [mockGameShelfEdge()],
        },
      });

      expect(mockFocusableCategoryCard).toHaveBeenCalledTimes(2);
      expect(mockFocusableStreamCard).not.toHaveBeenCalled();
    });

    it('renders a FocusableStreamCard for each Stream content node', () => {
      setup({
        shelves: {
          edges: [mockStreamShelfEdge()],
        },
      });

      expect(mockFocusableCategoryCard).not.toHaveBeenCalled();
      expect(mockFocusableStreamCard).toHaveBeenCalledTimes(2);
    });
  });

  describe('partial rendering', () => {
    it('renders internal error when when all shelves are empty', () => {
      const { wrapper } = setup({ shelves: { edges: [] } });

      expect(wrapper.find(InternalServerError)).toExist();
    });
  });
  describe(Homepage.getInitialProps!, () => {
    it('sets RecommendationShelves arguments', async () => {
      const inputPlatform = Platform.StarshotDev;
      const {
        queryVariables: {
          itemsPerRow,
          limit,
          platform: receivedPlatform,
          requestID,
        },
      } = await Homepage.getInitialProps!({
        platform: inputPlatform,
      } as any);

      expect(itemsPerRow).toEqual(RECOMMENDATION_ITEMS_PER_ROW);
      expect(limit).toEqual(RECOMMENDATION_NUM_OF_SHELVES);
      expect(receivedPlatform).toEqual(inputPlatform);
      expect(requestID).toHaveLength(ID_LENGTH);
    });
  });
});
