import type { WithRecommendationTracking } from 'tachyon-discovery';
import { ContentType, withRecommendationTracking } from 'tachyon-discovery';
import type { IsolateGqlUnionType, OmitRefType } from 'tachyon-type-library';
import type { TrackingNode } from 'tachyon-utils';
import {
  reduceToNonNullNodes,
  reduceToNonNullNodesWithTrackingID,
} from 'tachyon-utils';
import type { Homepage_card } from './__generated__/Homepage_card.graphql';
import type { Homepage_shelf } from './__generated__/Homepage_shelf.graphql';
import type { Homepage_shelves } from './__generated__/Homepage_shelves.graphql';
import type { Homepage_stream } from './__generated__/Homepage_stream.graphql';

type Card = TrackingNode<OmitRefType<Homepage_card>>;
export type StreamOrGameCard =
  | IsolateGqlUnionType<'Game', Card>
  | IsolateGqlUnionType<'Stream', Card>;
type RecommendationShelf = OmitRefType<Homepage_shelf> & {
  cards: Array<WithRecommendationTracking<StreamOrGameCard>>;
};
export type Stream = OmitRefType<Homepage_stream>;

type Shelf = OmitRefType<Homepage_shelves>;

function validCardType(card: Card): card is StreamOrGameCard {
  return ['Game', 'Stream'].includes(card.__typename);
}

export function getRecommendationShelves(
  shelves: Shelf | null,
): RecommendationShelf[] {
  return reduceToNonNullNodes(shelves?.edges)
    .map((shelf, shelfIdx) => {
      const cards = reduceToNonNullNodesWithTrackingID(shelf.content.edges)
        .filter(validCardType)
        .map((card, idx) => {
          const contentType =
            card.__typename === 'Game'
              ? ContentType.Game
              : ContentType.LiveStream;

          return withRecommendationTracking({
            card,
            cardIdx: idx,
            contentType,
            shelfIdx,
            tracking: shelf.trackingInfo,
          });
        });

      return {
        cards,
        id: shelf.id,
        title: shelf.title,
      };
    })
    .filter((shelf) => shelf.cards.length > 0);
}

export function shelfType(shelf: RecommendationShelf): 'Game' | 'Stream' {
  return shelf.cards[0].__typename;
}

export function extractStream(card: StreamOrGameCard): Stream | null {
  if (card.__typename === 'Game') {
    return card.streams?.edges?.[0]?.node ?? null;
  }
  return card;
}
