import { useRouter } from 'next/router';
import styled from 'styled-components';
import { useDeviceCodeFlowAuth } from 'tachyon-auth';
import { useIntl } from 'tachyon-intl';
import { defaultPageviewTracking, getCurrentUser } from 'tachyon-page-utils';
import { flattenHeaderOrParam } from 'tachyon-utils';
import {
  CoreText,
  Display,
  FlexDirection,
  Layout,
  TextType,
} from 'twitch-core-ui';
import { DarkThemeMap } from 'twitch-core-ui-tokens';
import { CLIENT_ID, REDIRECT_ROUTE_PARAM_KEY } from '../../../config';
import {
  Redirect,
  RouteName,
  routeAndRouteParamsFromQueryString,
} from '../../../routing';
import { FullScreenLoadingContent, QRLayout, StarshotMain } from '../../common';
import type { StarshotPage } from '../types';

const BOX_PADDING_REM = 5;
const OR_ELEMENT_SIZE_REM = 6;

const ScOr = styled.div`
  background: ${DarkThemeMap['color-background-float']};
  border-radius: 100%;
  bottom: 0;
  font-size: 1.5rem;
  height: ${OR_ELEMENT_SIZE_REM}rem;
  left: 50%;
  line-height: ${OR_ELEMENT_SIZE_REM}rem;
  position: absolute;
  right: 50%;
  text-align: center;
  transform: translate(-${BOX_PADDING_REM}rem, ${OR_ELEMENT_SIZE_REM / 2}rem);
  width: ${OR_ELEMENT_SIZE_REM}rem;
`;

const ScContentBoxTop = styled.div`
  background: ${DarkThemeMap['color-background-accent']};
  flex-grow: 1;
  height: 50%;
  padding: ${BOX_PADDING_REM}rem;
  padding-right: ${BOX_PADDING_REM * 2}rem;
  position: relative;
  text-align: center;
`;

const ScContentBoxBottom = styled.div`
  background: ${DarkThemeMap['color-background-base']};
  flex-grow: 1;
  height: 50%;
  padding: ${BOX_PADDING_REM}rem;
  padding-right: ${BOX_PADDING_REM * 2}rem;
  text-align: center;
`;

const ScCodePrompt = styled(CoreText)`
  padding: 0 6rem 2rem;
`;

const ScQRPrompt = styled(CoreText)`
  padding-top: 2rem;
`;

export const Login: StarshotPage = () => {
  const { query } = useRouter();
  const successRedirect = routeAndRouteParamsFromQueryString(
    flattenHeaderOrParam(query[REDIRECT_ROUTE_PARAM_KEY]),
  ) ?? {
    route: RouteName.Homepage,
    routeParams: undefined,
  };
  const { formatMessage } = useIntl();
  const dcf = useDeviceCodeFlowAuth(CLIENT_ID);

  switch (dcf.status) {
    case 'DEVICE_CODE_LOADING': {
      return <FullScreenLoadingContent />;
    }
    case 'DEVICE_CODE_FAIL': {
      // throw so we go to the error page and don't end up in loop
      // when there is an interpol service or network issue
      throw dcf.value;
    }
    case 'LOGGED_IN':
    case 'LOGIN_SUCCESS': {
      return <Redirect params={successRedirect} replace />;
    }
    case 'LOGIN_POLLING': {
      const code = dcf.value;
      return (
        <StarshotMain>
          <QRLayout content="login_page" url={code.qrCodeUri}>
            <Layout
              display={Display.Flex}
              flexDirection={FlexDirection.Column}
              fullHeight
              fullWidth
            >
              <ScContentBoxTop>
                <CoreText type={TextType.H4}>
                  {formatMessage('Join the Twitch community!', 'LoginPage')}
                </CoreText>
                <ScQRPrompt type={TextType.H3}>
                  {formatMessage(
                    'Scan the QR code on the right with your mobile device...',
                    'LoginPage',
                  )}
                </ScQRPrompt>
                <ScOr>{formatMessage('OR', 'LoginPage')}</ScOr>
              </ScContentBoxTop>
              <ScContentBoxBottom>
                <ScCodePrompt type={TextType.H4}>
                  {formatMessage(
                    'visit {verificationUri} and enter this {length} digit code',
                    {
                      length: code.userCode.length,
                      verificationUri: code.verificationUri,
                    },
                    'LoginPage',
                  )}
                </ScCodePrompt>
                <CoreText type={TextType.H1}>{code.formattedUserCode}</CoreText>
              </ScContentBoxBottom>
            </Layout>
          </QRLayout>
        </StarshotMain>
      );
    }
    default: {
      const exhausted: never = dcf;
      return exhausted;
    }
  }
};

Login.currentUser = getCurrentUser;
Login.displayName = 'Login';
Login.navigationBehavior = () => ({
  displayBackButton: true,
  displayNavMenu: true,
  noFocusableContent: true,
});
Login.pageviewTracking = defaultPageviewTracking;
