import { useRouter } from 'next/router';
import { useDeviceCodeFlowAuth } from 'tachyon-auth';
import { createMountWrapperFactory } from 'tachyon-test-utils';
import { RouteName, createRedirectQuery } from '../../../routing';
import { mockRouter } from '../../../routing/test-mocks';
import { LoadingContent } from '../../common';
import { Login } from '.';

jest.mock('tachyon-auth', () => ({
  ...jest.requireActual('tachyon-auth'),
  useDeviceCodeFlowAuth: jest.fn(),
}));

const mockUseDeviceCodeFlowAuth = useDeviceCodeFlowAuth as jest.Mock;

jest.mock('next/router', () => ({
  ...jest.requireActual('next/router'),
  useRouter: jest.fn(() => ({ query: {} })),
}));

const mockUseRouter = useRouter as jest.Mock;

jest.mock('../../../routing/useTachyonRouter', () => ({
  useTachyonRouter: () => mockRouter,
}));

jest.mock('tachyon-experiments', () => {
  const original = jest.requireActual('tachyon-experiments');
  return {
    ...original,
    Experiment: jest.fn(original.MockExperimentControl),
  };
});

describe(Login, () => {
  const setup = createMountWrapperFactory(Login);

  it('redirects when the user is already logged in', () => {
    mockUseDeviceCodeFlowAuth.mockImplementationOnce(() => ({
      status: 'LOGGED_IN',
    }));

    setup();

    expect(mockRouter.replace).toHaveBeenCalledWith(
      {
        route: RouteName.Homepage,
        routeParams: undefined,
      },
      undefined,
    );
  });

  describe('device code', () => {
    it('renders loading when fetching', () => {
      mockUseDeviceCodeFlowAuth.mockImplementationOnce(() => ({
        status: 'DEVICE_CODE_LOADING',
      }));

      const { wrapper } = setup();

      expect(wrapper.find(LoadingContent)).toExist();
    });

    it('renders QR instructions when present', () => {
      mockUseDeviceCodeFlowAuth.mockImplementationOnce(() => ({
        status: 'LOGIN_POLLING',
        value: {
          formattedUserCode: 'WDJB-MJHT',
          qrCodeUri:
            'https://www.twitch.tv/?device-code=WDJBMJHT&device-metadata=qr_code',
          userCode: 'WDJBMJHT',
          verificationUri: 'https://www.twitch.tv/activate',
        },
      }));

      const { wrapper } = setup();

      expect(wrapper).toIncludeText('WDJB-MJHT');
    });

    it('throws when fetch fails', () => {
      const error = new Error('test uncaught error');

      // Not sure why the hook is invoked twice for error case
      mockUseDeviceCodeFlowAuth
        .mockImplementationOnce(() => ({
          status: 'DEVICE_CODE_FAIL',
          value: error,
        }))
        .mockImplementationOnce(() => ({
          status: 'DEVICE_CODE_FAIL',
          value: error,
        }));

      expect(setup).toThrow(error);
    });
  });

  describe('success redirect', () => {
    beforeEach(() => {
      mockUseDeviceCodeFlowAuth.mockImplementationOnce(() => ({
        status: 'LOGIN_SUCCESS',
      }));
    });

    it('redirects to the "redirect" prop (and replaces the history) upon logging in', () => {
      setup();

      expect(mockRouter.replace).toHaveBeenCalledWith(
        {
          route: RouteName.Homepage,
          routeParams: undefined,
        },
        undefined,
      );
    });

    it('extracts redirect from query params', () => {
      mockUseRouter.mockImplementationOnce(() => ({
        query: createRedirectQuery({
          route: RouteName.Channel,
          routeParams: {
            login: 'therealderekt',
          },
        }),
      }));

      setup();

      expect(mockRouter.replace).toHaveBeenCalledWith(
        {
          route: RouteName.Channel,
          routeParams: {
            login: 'therealderekt',
          },
        },
        undefined,
      );
    });

    it('falls back to Homepage if redirect is missing', () => {
      setup();

      expect(mockRouter.replace).toHaveBeenCalledWith(
        {
          route: RouteName.Homepage,
          routeParams: undefined,
        },
        undefined,
      );
    });
  });
});
