import type { FC } from 'react';
import { SearchQueryType, useDiscoveryTracking } from 'tachyon-discovery';
import { useIntl } from 'tachyon-intl';
import { VerticalNav } from 'tachyon-tv-nav';
import {
  CoreText,
  FontSize,
  FontWeight,
  Layout,
  SVGAsset,
  TextTransform,
  TextType,
} from 'twitch-core-ui';
import { SEARCH_TERM_QUERY_PARAM } from '../../../../config';
import { useSearchHistory } from '../../../../hooks';
import {
  RouteName,
  renderTachyonLink,
  useTachyonRouter,
} from '../../../../routing';
import { FocusableTextButton } from '../../../common';
import { SearchInput } from '../SearchInput';

type SearchDropMenuProps = {
  focusIndex: number;
};

export const HISTORY_DISPLAY_LIMIT = 5;

export const SearchInputWithHistory: FC<SearchDropMenuProps> = ({
  focusIndex,
}) => {
  const { formatMessage } = useIntl();
  const router = useTachyonRouter();
  const { clearHistory, history: allHistory, pushTerm } = useSearchHistory();
  const { onSearchInputFocus, onSearchQuerySubmit } = useDiscoveryTracking();

  function submitHandler(searchTerm: string): void {
    pushTerm(searchTerm);
    onSearchQuerySubmit({
      query: searchTerm,
      type: SearchQueryType.UserTyped,
    });

    router.push(
      {
        route: RouteName.Search,
      },
      {
        [SEARCH_TERM_QUERY_PARAM]: searchTerm,
      },
    );
  }

  const history = allHistory.slice(0, HISTORY_DISPLAY_LIMIT);

  const hasSearchHistory = history.length > 0;

  // history entries + clear history menu option
  const historyMenuItems = hasSearchHistory ? history.length + 1 : 0;
  // search input + history menu items
  const navigableElementCount = 1 + historyMenuItems;

  return (
    <VerticalNav
      elementCount={navigableElementCount}
      focusIndex={focusIndex}
      takeFocusOnFirstRender
    >
      <SearchInput
        focusIndex={0}
        onFocus={onSearchInputFocus}
        onSubmit={submitHandler}
      />
      {hasSearchHistory && (
        <>
          <Layout padding={{ bottom: 2, top: 3 }}>
            <CoreText
              fontSize={FontSize.Size6}
              fontWeight={FontWeight.Regular}
              transform={TextTransform.Uppercase}
              type={TextType.H3}
            >
              {formatMessage('Recent Searches:', 'SearchDropMenu')}
            </CoreText>
          </Layout>
          {history.map((searchTerm, index) => (
            <Layout key={searchTerm} padding={{ bottom: 1 }}>
              <FocusableTextButton
                contentLeft
                focusIndex={1 + index}
                fullWidth
                icon={SVGAsset.SearchHistory}
                linkTo="/deferToRenderLink"
                onClick={() => {
                  pushTerm(searchTerm);
                  onSearchQuerySubmit({
                    query: searchTerm,
                    type: SearchQueryType.History,
                  });
                }}
                renderLink={renderTachyonLink({
                  query: {
                    [SEARCH_TERM_QUERY_PARAM]: searchTerm,
                  },
                  route: RouteName.Search,
                })}
              >
                <CoreText fontSize={FontSize.Size5} type={TextType.H4}>
                  {searchTerm}
                </CoreText>
              </FocusableTextButton>
            </Layout>
          ))}
          <FocusableTextButton
            contentLeft
            focusIndex={navigableElementCount - 1}
            fullWidth
            icon={SVGAsset.Trash}
            onClick={clearHistory}
          >
            <CoreText fontSize={FontSize.Size5}>
              {formatMessage('Clear Recent Searches', 'SearchDropMenu')}
            </CoreText>
          </FocusableTextButton>
        </>
      )}
    </VerticalNav>
  );
};

SearchInputWithHistory.displayName = 'SearchInputWithHistory';
