import { createMountWrapperFactory } from 'tachyon-test-utils';
import { VerticalNav } from 'tachyon-tv-nav';
import { DropDownMenuHeading } from 'twitch-core-ui';
import { SEARCH_TERM_QUERY_PARAM } from '../../../../config';
import { RouteName } from '../../../../routing';
import { FocusableTextButton } from '../../../common';
import { SearchInput } from '../SearchInput';
import { HISTORY_DISPLAY_LIMIT, SearchInputWithHistory } from '.';

const mockTachyonRouter = { push: jest.fn() };
jest.mock('../../../../routing', () => ({
  ...jest.requireActual('../../../../routing'),
  useTachyonRouter: () => mockTachyonRouter,
}));

jest.mock('../SearchInput', () => ({
  SearchInput: jest.fn(() => <div />),
}));

let mockSearchHistory: string[] = [];
const mockPushSearchTerm = jest.fn();
const mockClearHistory = jest.fn();

jest.mock('../../../../hooks', () => ({
  useSearchHistory: () => ({
    clearHistory: mockClearHistory,
    history: mockSearchHistory,
    pushTerm: mockPushSearchTerm,
  }),
}));

describe(SearchInputWithHistory, () => {
  const setup = createMountWrapperFactory(SearchInputWithHistory, () => ({
    focusIndex: 0,
  }));

  beforeEach(() => {
    mockSearchHistory = [];
  });

  it('pushes search history and to the router on search input', () => {
    const { wrapper } = setup();
    const searchTerm = 'therealderekt';

    wrapper.find(SearchInput).prop('onSubmit')(searchTerm);

    expect(mockPushSearchTerm).toHaveBeenCalledTimes(1);
    expect(mockPushSearchTerm).toHaveBeenCalledWith(searchTerm);

    expect(mockTachyonRouter.push).toHaveBeenCalledTimes(1);
    expect(mockTachyonRouter.push).toHaveBeenCalledWith(
      {
        route: RouteName.Search,
      },
      {
        [SEARCH_TERM_QUERY_PARAM]: searchTerm,
      },
    );
  });

  describe('when the user has no search history', () => {
    beforeEach(() => {
      mockSearchHistory = [];
    });

    it('specifies the correct focus count for only the input', () => {
      const { wrapper } = setup();
      expect(wrapper.find(VerticalNav)).toHaveProp({ elementCount: 1 });
    });

    it('does not render the history area', () => {
      const { wrapper } = setup();
      expect(wrapper.find(DropDownMenuHeading)).not.toExist();
    });
  });

  describe('when the user has a search history', () => {
    beforeEach(() => {
      mockSearchHistory = ['Fortnite', 'Tfue', 'Rocket League Hits'];
    });

    it('renders each of the provided history items + the clear option', () => {
      const { wrapper } = setup();
      expect(wrapper.find(FocusableTextButton)).toHaveLength(
        mockSearchHistory.length + 1,
      );

      expect(wrapper.find(VerticalNav)).toHaveProp({
        // input, history items, clear
        elementCount: 2 + mockSearchHistory.length,
      });
    });

    it(`does not render more than ${HISTORY_DISPLAY_LIMIT} history items`, () => {
      mockSearchHistory = [
        'Fortnite',
        'Tfue',
        'Rocket League Hits',
        'Lirik',
        'DansGaming',
        'therealderekt',
      ];

      const { wrapper } = setup();

      expect(wrapper.find(VerticalNav)).toHaveProp({
        // input, history items, clear
        elementCount: HISTORY_DISPLAY_LIMIT + 2,
      });
    });

    it('updates the history order when user clicks a recent search term', () => {
      const { wrapper } = setup();
      wrapper.find(FocusableTextButton).at(0).prop('onClick')!({} as any);

      expect(mockPushSearchTerm).toHaveBeenCalledWith(mockSearchHistory[0]);
    });

    it('clears history when the user clicks the option', () => {
      const { wrapper } = setup();
      wrapper.find(FocusableTextButton).last().prop('onClick')!({} as any);

      expect(mockClearHistory).toHaveBeenCalledTimes(1);
    });
  });
});
