import { act } from '@testing-library/react-hooks';
import { datatype, random } from 'faker';
import { createMountWrapperFactory } from 'tachyon-test-utils';
import {
  FocusableOfflineChannelCard,
  FocusableStreamCard,
} from '../../../../common';
import { FocusableSearchChannelCardBase as FocusableSearchChannelCard } from '.';

jest.mock('../../../../common', () => ({
  FocusableOfflineChannelCard: jest.fn(() => <div />),
  FocusableStreamCard: jest.fn(() => <div />),
}));

describe(FocusableSearchChannelCard, () => {
  const setup = createMountWrapperFactory(FocusableSearchChannelCard, () => ({
    cardIndex: datatype.number(),
    isRelatedContent: false,
    node: {
      ' $fragmentRefs': {
        FocusableOfflineChannelCard_channel: true,
      },
      bannerImageURL: '',
      id: datatype.uuid(),
      stream: null,
    },
    onFocus: jest.fn(),
    onSearchImpressionMemoized: jest.fn(),
    onSearchResultClick: jest.fn(),
    rowPosition: datatype.number(),
    searchTerm: random.alphaNumeric(),
  }));

  describe('renders', () => {
    it('fires onSearchImpression', () => {
      const { props } = setup();
      expect(props.onSearchImpressionMemoized).toHaveBeenCalledTimes(1);
    });

    it('a channel card when channel is offline', () => {
      const { props, wrapper } = setup();
      const channelCard = wrapper.find(FocusableOfflineChannelCard);

      expect(channelCard).toExist();
      expect(channelCard).toHaveProp('focusIndex', props.cardIndex);
    });

    it('a stream card when a channel is provided', () => {
      const { props, wrapper } = setup({
        node: { stream: { id: datatype.uuid() } },
      });
      const streamCard = wrapper.find(FocusableStreamCard);

      expect(streamCard).toExist();
      expect(streamCard).toHaveProp('focusIndex', props.cardIndex);
    });
  });

  it('calls setBackgroundImage on focus', () => {
    const { props, wrapper } = setup();
    expect(props.onFocus).not.toHaveBeenCalled();

    act(() => {
      wrapper.find(FocusableOfflineChannelCard).prop('onFocus')!(
        props.cardIndex,
      );
    });

    expect(props.onFocus).toHaveBeenCalledWith(props.cardIndex);
  });

  it('calls onSearchResultClick when clicked', () => {
    const { props, wrapper } = setup();
    expect(props.onSearchResultClick).not.toHaveBeenCalled();

    act(() => {
      wrapper.find(FocusableOfflineChannelCard).prop('onClick')!();
    });

    expect(props.onSearchResultClick).toHaveBeenCalledTimes(1);
  });
});
