import type { FC } from 'react';
import { useCallback, useMemo } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import { useDiscoveryTracking } from 'tachyon-discovery';
import { useIntl } from 'tachyon-intl';
import { isValidObject } from 'tachyon-relay';
import { useCallbackMemoizedByKey } from 'tachyon-utils';
import { TextType, Title, TitleSize, WordBreak } from 'twitch-core-ui';
import {
  CATEGORY_CARD_CONFIG,
  WATCHABLE_CARD_CONFIG,
} from '../../../../config';
import type { HorizontalShelfProps } from '../../../common';
import {
  BannerPage,
  HorizontalShelf,
  VerticalShelfList,
  useBannerData,
} from '../../../common';
import { SearchCTA } from '../SearchCTA';
import { channelBackgroundImageSrc } from '../utils';
import { FocusableSearchCategoryCard } from './FocusableSearchCategoryCard';
import { FocusableSearchChannelCard } from './FocusableSearchChannelCard';
import { FocusableSearchVideoCard } from './FocusableSearchVideoCard';
import { SearchBannerContent } from './SearchBannerContent';
import type { SearchResults_results } from './__generated__/SearchResults_results.graphql';

export type { SearchResults_results } from './__generated__/SearchResults_results.graphql';

interface SearchResultsProps {
  focusIndex: number;
  results: SearchResults_results;
  searchTerm: string;
}

type SearchResult<T extends Exclude<keyof SearchResults_results, ' $refType'>> =
  NonNullable<NonNullable<SearchResults_results[T]>['items']>[0];

type StreamSearchResult = SearchResult<'channels'>;
type GameSearchResult = SearchResult<'games'>;
type VideoSearchResult = SearchResult<'videos'>;

type Shelf = FC<
  Pick<
    HorizontalShelfProps<
      GameSearchResult | StreamSearchResult | VideoSearchResult
    >,
    'focusIndex'
  >
>;

export const SearchResultsBase: FC<SearchResultsProps> = ({
  focusIndex,
  results,
  searchTerm,
}) => {
  const { formatMessage } = useIntl();
  const { onSearchResultClick, onSearchResultImpression } =
    useDiscoveryTracking();

  const onSearchImpressionMemoized = useCallbackMemoizedByKey(
    onSearchResultImpression,
    {
      targetKeys: ['contentID'],
    },
  );

  // This needs to be memo'd to prevent each of the filtered result sets from being
  // re-computed on every render which would cause each shelf to have to be reconciled
  // as well
  const { channels, games, initialBackgroundImage, relatedChannels, videos } =
    useMemo(() => {
      const validChannels = (results.channels?.items ?? []).filter(
        isValidObject,
      );
      const validGames = (results.games?.items ?? []).filter(isValidObject);
      const validRelatedChannels = (
        results.relatedLiveChannels?.items ?? []
      ).filter(isValidObject);
      const validVideos = (results.videos?.items ?? []).filter(isValidObject);

      let bgImg: string | null = null;
      if (validChannels.length > 0) {
        bgImg = channelBackgroundImageSrc(validChannels[0]);
      } else if (validGames.length > 0) {
        bgImg =
          validGames[0].streams?.edges?.[0]?.node?.previewImageURL ?? null;
      } else if (validRelatedChannels.length > 0) {
        bgImg = channelBackgroundImageSrc(validRelatedChannels[0]);
      } else if (validVideos.length > 0) {
        bgImg = validVideos[0].previewThumbnailURL;
      }

      return {
        channels: validChannels,
        games: validGames,
        initialBackgroundImage: bgImg,
        relatedChannels: validRelatedChannels,
        videos: validVideos,
      };
    }, [results]);

  const updateBannerData = useBannerData<string>({
    backgroundImageSrc: initialBackgroundImage,
    contentData: searchTerm,
  });

  const onChannelCardFocus = useCallback(
    (index: number) => {
      updateBannerData({
        backgroundImageSrc: channelBackgroundImageSrc(channels[index]),
        contentData: searchTerm,
      });
    },
    [channels, searchTerm, updateBannerData],
  );

  const onRelatedFocus = useCallback(
    (index: number) => {
      updateBannerData({
        backgroundImageSrc: channelBackgroundImageSrc(relatedChannels[index]),
        contentData: searchTerm,
      });
    },
    [relatedChannels, searchTerm, updateBannerData],
  );

  const onCategoryCardFocus = useCallback(
    (index: number) => {
      updateBannerData({
        backgroundImageSrc:
          games[index].streams?.edges?.[0]?.node?.previewImageURL ?? null,
        contentData: searchTerm,
      });
    },
    [games, searchTerm, updateBannerData],
  );

  const onVideoCardFocus = useCallback(
    (index: number) => {
      updateBannerData({
        backgroundImageSrc: videos[index].previewThumbnailURL,
        contentData: searchTerm,
      });
    },
    [videos, searchTerm, updateBannerData],
  );

  // The shelves result needs to be memoized, otherwise a new array would be generated
  // on each render which React cannot shallow compare.
  const shelves = useMemo(() => {
    const shelvesToRender: Array<Shelf> = [];

    if (channels.length > 0) {
      shelvesToRender.push(({ focusIndex: shelfFocusIndex }) => (
        <HorizontalShelf
          config={WATCHABLE_CARD_CONFIG}
          focusIndex={shelfFocusIndex}
          itemRenderer={(node, cardIndex) => (
            <FocusableSearchChannelCard
              cardIndex={cardIndex}
              isRelatedContent={false}
              key={cardIndex}
              node={node}
              onFocus={onChannelCardFocus}
              onSearchImpressionMemoized={onSearchImpressionMemoized}
              onSearchResultClick={onSearchResultClick}
              rowPosition={shelvesToRender.length - 1}
              searchTerm={searchTerm}
            />
          )}
          items={channels}
          title={formatMessage('Channels', 'SearchResults')}
          virtualized
        />
      ));
    }

    if (relatedChannels.length > 0) {
      shelvesToRender.push(({ focusIndex: shelfFocusIndex }) => (
        <HorizontalShelf
          config={WATCHABLE_CARD_CONFIG}
          focusIndex={shelfFocusIndex}
          itemRenderer={(node, cardIndex) => (
            <FocusableSearchChannelCard
              cardIndex={cardIndex}
              isRelatedContent
              key={cardIndex}
              node={node}
              onFocus={onRelatedFocus}
              onSearchImpressionMemoized={onSearchImpressionMemoized}
              onSearchResultClick={onSearchResultClick}
              rowPosition={shelvesToRender.length - 1}
              searchTerm={searchTerm}
            />
          )}
          items={relatedChannels}
          title={formatMessage(
            'People searching for "{searchTerm}" also watch',
            { searchTerm },
            'SearchResults',
          )}
          virtualized
        />
      ));
    }

    if (games.length > 0) {
      shelvesToRender.push(({ focusIndex: shelfFocusIndex }) => (
        <HorizontalShelf
          config={CATEGORY_CARD_CONFIG}
          focusIndex={shelfFocusIndex}
          itemRenderer={(node, cardIndex) => (
            <FocusableSearchCategoryCard
              cardIndex={cardIndex}
              key={cardIndex}
              node={node}
              onFocus={onCategoryCardFocus}
              onSearchImpressionMemoized={onSearchImpressionMemoized}
              onSearchResultClick={onSearchResultClick}
              rowPosition={shelvesToRender.length - 1}
              searchTerm={searchTerm}
            />
          )}
          items={games}
          title={formatMessage('Games', 'SearchResults')}
          virtualized
        />
      ));
    }

    if (videos.length > 0) {
      shelvesToRender.push(({ focusIndex: shelfFocusIndex }) => (
        <HorizontalShelf
          config={WATCHABLE_CARD_CONFIG}
          focusIndex={shelfFocusIndex}
          itemRenderer={(node, cardIndex) => (
            <FocusableSearchVideoCard
              cardIndex={cardIndex}
              key={cardIndex}
              node={node}
              onFocus={onVideoCardFocus}
              onSearchImpressionMemoized={onSearchImpressionMemoized}
              onSearchResultClick={onSearchResultClick}
              rowPosition={shelvesToRender.length - 1}
              searchTerm={searchTerm}
            />
          )}
          items={videos}
          title={formatMessage('Past Videos', 'SearchResults')}
          virtualized
        />
      ));
    }

    return shelvesToRender;
  }, [
    channels,
    games,
    formatMessage,
    onSearchResultClick,
    searchTerm,
    onChannelCardFocus,
    onCategoryCardFocus,
    onRelatedFocus,
    onSearchImpressionMemoized,
    onVideoCardFocus,
    relatedChannels,
    videos,
  ]);

  const bannerContentProps = useMemo(
    () => ({
      searchTerm,
    }),
    [searchTerm],
  );

  if (shelves.length === 0) {
    return (
      <SearchCTA focusIndex={focusIndex}>
        <Title
          size={TitleSize.Default}
          type={TextType.H2}
          wordBreak={WordBreak.BreakWord}
        >
          {formatMessage(
            'No live results for: {searchTerm}.',
            { searchTerm },
            'SearchPage',
          )}
        </Title>
      </SearchCTA>
    );
  }

  return (
    <BannerPage
      BannerContent={SearchBannerContent}
      bannerContentProps={bannerContentProps}
      updateBannerData={updateBannerData}
    >
      <VerticalShelfList
        focusIndex={focusIndex}
        items={shelves}
        shelfRenderer={(Shelf, index) => (
          <Shelf focusIndex={index} key={index} />
        )}
        takeFocusOnFirstRender
      />
    </BannerPage>
  );
};

SearchResultsBase.displayName = 'SearchResultShelvesBase';

// eslint-disable-next-line no-unused-expressions
graphql`
  fragment SearchResults_channelsItem on User {
    id
    bannerImageURL
    ...FocusableOfflineChannelCard_channel
    stream {
      id
      previewImageURL
      ...FocusableStreamCard_stream
      game {
        id
      }
    }
  }
`;

// eslint-disable-next-line no-unused-expressions
graphql`
  fragment SearchResults_gamesItem on Game {
    id
    ...FocusableCategoryCard_category
    streams(first: 1) {
      edges {
        node {
          previewImageURL
          id
        }
      }
    }
  }
`;

// eslint-disable-next-line no-unused-expressions
graphql`
  fragment SearchResults_videosItem on Video {
    ...FocusableVodCard_video
    id
    game {
      id
    }
    previewThumbnailURL
  }
`;

// We unmask "SearchGameShelf_games" for partial success handling due to
// https://github.com/facebook/relay/issues/3041
export const SearchResults = createFragmentContainer(SearchResultsBase, {
  results: graphql`
    fragment SearchResults_results on SearchFor {
      channels {
        items {
          ...SearchResults_channelsItem @relay(mask: false)
        }
      }
      relatedLiveChannels {
        items {
          ...SearchResults_channelsItem @relay(mask: false)
        }
      }
      games {
        items {
          ...SearchResults_gamesItem @relay(mask: false)
        }
      }
      videos {
        items {
          ...SearchResults_videosItem @relay(mask: false)
        }
      }
    }
  `,
});

SearchResults.displayName = 'SearchResultShelves';
