import { datatype, image, internet, lorem } from 'faker';
import { SearchCategory } from 'tachyon-discovery';
import { convertToUnsafeID, garbageId, validId } from 'tachyon-relay';
import { createMountWrapperFactory } from 'tachyon-test-utils';
import { VerticalShelfList, useBannerData } from '../../../common';
import { SearchCTA } from '../SearchCTA';
import type { Channel, Game, LiveStream, Video } from '../utils';
import {
  getGameTrackingParams,
  getStreamTrackingParams,
  getVideoTrackingParams,
} from './utils';
import { SearchResultsBase } from '.';

jest.mock('../../../common', () => {
  const { MockHorizontalShelf } = jest.requireActual(
    '../../../common/HorizontalShelf/test-mocks',
  );
  const { MockVerticalShelfList } = jest.requireActual(
    '../../../common/VerticalShelfList/test-mocks',
  );
  return {
    ...jest.requireActual('../../../common'),
    FocusableCategoryCard: jest.fn(() => <div />),
    FocusableOfflineChannelCard: jest.fn(() => <div />),
    FocusableStreamCard: jest.fn(() => <div />),
    FocusableVodCard: jest.fn(() => <div />),
    HorizontalShelf: MockHorizontalShelf,
    VerticalShelfList: MockVerticalShelfList,
    useBannerData: jest.fn(),
  };
});
const mockUseBannerData = useBannerData as jest.Mock;

function mockStream(): LiveStream {
  return {
    ' $fragmentRefs': {
      FocusableStreamCard_stream: true,
    },
    game: {
      id: datatype.uuid(),
    },
    id: validId(datatype.uuid()),
    previewImageURL: image.imageUrl(),
  };
}

function mockChannel(overrides?: Partial<Channel>): Channel {
  return {
    ' $fragmentRefs': {
      FocusableOfflineChannelCard_channel: true,
    },
    bannerImageURL: image.imageUrl(),
    id: validId(datatype.uuid()),
    stream: mockStream(),
    ...overrides,
  };
}

function mockGame(overrides?: Partial<Game>): Game {
  return {
    ' $fragmentRefs': {
      FocusableCategoryCard_category: true,
    },
    id: validId(datatype.uuid()),
    streams: {
      edges: [
        {
          node: {
            id: datatype.uuid(),
            previewImageURL: image.imageUrl(),
          },
        },
      ],
    },
    ...overrides,
  };
}

function mockVideo(overrides?: Partial<Video>): Video {
  return {
    ' $fragmentRefs': {
      FocusableVodCard_video: true,
    },
    game: {
      id: validId(datatype.uuid()),
    },
    id: validId(datatype.uuid()),
    previewThumbnailURL: image.imageUrl(),
    ...overrides,
  };
}

const ChannelShelf = { title: 'Channels' };
const GameShelf = { title: 'Games' };
const RelatedLiveChannelsShelf = {
  title: 'People searching for "Some search term" also watch',
};
const VodsShelf = { title: 'Past Videos' };

describe(SearchResultsBase, () => {
  const setup = createMountWrapperFactory(SearchResultsBase, () => ({
    focusIndex: 0,
    onCardFocus: jest.fn(),
    results: {
      ' $refType': 'SearchResults_results',
      channels: {
        items: [mockChannel(), mockChannel({ stream: null })],
      },
      games: {
        items: [mockGame()],
      },
      relatedLiveChannels: {
        items: [mockChannel()],
      },
      videos: {
        items: [mockVideo(), mockVideo()],
      },
    },
    searchTerm: 'Some search term',
  }));

  it('sets the initial background image', () => {
    const { props } = setup();

    expect(mockUseBannerData).toHaveBeenCalledWith({
      backgroundImageSrc:
        props.results.channels!.items![0].stream!.previewImageURL,
      contentData: props.searchTerm,
    });
  });

  describe('shelves', () => {
    it('renders the no results message for an empty result set', () => {
      const { wrapper } = setup({
        results: {
          channels: { items: [] },
          games: { items: [] },
          relatedLiveChannels: { items: [] },
          videos: { items: [] },
        },
      });

      expect(wrapper.find(SearchCTA)).toExist();
      expect(wrapper.find(VerticalShelfList)).not.toExist();
    });

    it('renders just a game shelf which takes focus', () => {
      const { wrapper } = setup({
        results: {
          channels: { items: [] },
          games: { items: [mockGame()] },
          relatedLiveChannels: { items: [] },
          videos: { items: [] },
        },
      });

      expect(wrapper.find(VerticalShelfList).prop('items')).toHaveLength(1);
      expect(wrapper.find(GameShelf).prop('items')).toHaveLength(1);
    });

    it('renders just a channel shelf with live and offline channels which takes focus', () => {
      const { wrapper } = setup({
        results: {
          channels: {
            items: [
              // online
              mockChannel(),
              mockChannel(),
              // offline
              mockChannel({ stream: null }),
              mockChannel({ stream: { id: garbageId() } as any }),
            ],
          },
          games: { items: [] },
          relatedLiveChannels: { items: [] },
          videos: { items: [] },
        },
      });

      expect(wrapper.find(VerticalShelfList).prop('items')).toHaveLength(1);
      expect(wrapper.find(ChannelShelf).prop('items')).toHaveLength(4);
    });

    it('renders just a related live channels shelf which takes focus', () => {
      const { wrapper } = setup({
        results: {
          channels: { items: [] },
          games: { items: [] },
          relatedLiveChannels: {
            items: [mockChannel(), mockChannel()],
          },
          videos: { items: [] },
        },
      });

      expect(wrapper.find(VerticalShelfList).prop('items')).toHaveLength(1);
      expect(wrapper.find(RelatedLiveChannelsShelf).prop('items')).toHaveLength(
        2,
      );
    });

    it('renders just a vods shelf which takes focus', () => {
      const { wrapper } = setup({
        results: {
          channels: { items: [] },
          games: { items: [] },
          relatedLiveChannels: { items: [] },
          videos: {
            items: [mockVideo(), mockVideo()],
          },
        },
      });

      expect(wrapper.find(VerticalShelfList).prop('items')).toHaveLength(1);
      expect(wrapper.find(VodsShelf).prop('items')).toHaveLength(2);
    });
  });

  describe('game search results', () => {
    it('filters out invalid game results', () => {
      const { wrapper } = setup({
        results: {
          games: {
            items: [mockGame({ id: garbageId() }), mockGame()],
          },
        },
      });

      expect(wrapper.find(GameShelf).prop('items')).toHaveLength(1);
    });
  });
});

describe('utils', () => {
  describe('getStreamTrackingParams', () => {
    it('returns params for a channel', () => {
      const id = datatype.uuid();
      const unsafeID = convertToUnsafeID(id);
      const cardIdx = datatype.number();
      const itemRow = datatype.number();
      const searchTerm = lorem.word();
      const params = getStreamTrackingParams(
        {
          ' $fragmentRefs': { FocusableOfflineChannelCard_channel: true },
          bannerImageURL: internet.url(),
          id,
          stream: {
            ' $fragmentRefs': { FocusableStreamCard_stream: true },
            game: null,
            id: datatype.uuid(),
            previewImageURL: null,
          },
        },
        cardIdx,
        itemRow,
        searchTerm,
        false,
      );

      expect(params).toMatchObject({
        contentID: unsafeID,
        contentType: SearchCategory.Streams,
        itemIndex: cardIdx,
        itemRow,
        query: searchTerm,
        subSection: SearchCategory.Streams,
      });
    });

    it('returns params for a related channel', () => {
      const id = datatype.uuid();
      const unsafeID = convertToUnsafeID(id);
      const cardIdx = datatype.number();
      const itemRow = datatype.number();
      const searchTerm = lorem.word();
      const params = getStreamTrackingParams(
        {
          ' $fragmentRefs': { FocusableOfflineChannelCard_channel: true },
          bannerImageURL: internet.url(),
          id,
          stream: {
            ' $fragmentRefs': { FocusableStreamCard_stream: true },
            game: null,
            id: datatype.uuid(),
            previewImageURL: null,
          },
        },
        cardIdx,
        itemRow,
        searchTerm,
        true,
      );

      expect(params).toMatchObject({
        contentID: unsafeID,
        contentType: SearchCategory.Streams,
        itemIndex: cardIdx,
        itemRow,
        query: searchTerm,
        subSection: SearchCategory.RelatedChannels,
      });
    });
  });

  describe('getGameTrackingParams', () => {
    it('returns params for a game', () => {
      const id = datatype.uuid();
      const unsafeID = convertToUnsafeID(id);
      const cardIdx = datatype.number();
      const itemRow = datatype.number();
      const searchTerm = lorem.word();
      const params = getGameTrackingParams(
        {
          id,
        },
        cardIdx,
        itemRow,
        searchTerm,
      );

      expect(params).toMatchObject({
        contentID: unsafeID,
        contentType: SearchCategory.Games,
        itemIndex: cardIdx,
        itemRow,
        query: searchTerm,
        subSection: SearchCategory.Games,
      });
    });
  });

  describe('getVideoTrackingParams', () => {
    it('returns params for a vod', () => {
      const id = datatype.uuid();
      const unsafeID = convertToUnsafeID(id);
      const gameID = datatype.uuid();
      const unsafeGameID = convertToUnsafeID(gameID);
      const cardIdx = datatype.number();
      const itemRow = datatype.number();
      const searchTerm = lorem.word();
      const previewThumbnailURL = internet.url();
      const params = getVideoTrackingParams(
        {
          ' $fragmentRefs': { FocusableVodCard_video: true },
          game: {
            id: gameID,
          },
          id,
          previewThumbnailURL,
        },
        cardIdx,
        itemRow,
        searchTerm,
      );

      expect(params).toMatchObject({
        contentID: unsafeID,
        contentType: SearchCategory.Videos,
        gameID: unsafeGameID,
        itemIndex: cardIdx,
        itemRow,
        query: searchTerm,
        subSection: SearchCategory.Videos,
      });
    });
  });
});
