import { graphql } from 'react-relay/hooks';
import { defaultPageviewTracking, getCurrentUser } from 'tachyon-page-utils';
import { flattenHeaderOrParam } from 'tachyon-utils';
import { SEARCH_TERM_QUERY_PARAM } from '../../../config';
import { StarshotMain } from '../../common';
import { InternalServerError } from '../../errors';
import type { StarshotPage } from '../types';
import { SearchContent } from './SearchContent';
import type {
  Search_QueryResponse,
  Search_QueryVariables,
} from './__generated__/Search_Query.graphql';

// TODO: move to page-utils
export type PathParameters = {
  /**
   * "term" can technically be undefined since it is derived from a query param. Because we have no external navigation to this page, we set our types to always be defined since we expect in-app consumers to always provide it.
   */
  term: string;
};

type SearchInitialProps = {
  queryVariables: Required<Search_QueryVariables>;
};

export type SearchProps = Search_QueryResponse & SearchInitialProps;

export const Search: StarshotPage<SearchInitialProps, SearchProps> = ({
  queryVariables,
  searchFor,
}) => {
  const { noQuery, userQuery } = queryVariables;

  // This state is reachable due to the presence of the retry button
  // we will probably want a more elegant solution down the road
  // https://jira.twitch.com/browse/EMP-3271
  if (!noQuery && !searchFor) {
    return <InternalServerError />;
  }

  return (
    <StarshotMain>
      <SearchContent
        content={searchFor ?? null}
        focusIndex={0}
        searchTerm={userQuery}
      />
    </StarshotMain>
  );
};

Search.currentUser = getCurrentUser;
Search.displayName = 'Search';
Search.navigationBehavior = () => ({ displayNavMenu: true });
Search.pageviewTracking = defaultPageviewTracking;

// TODO: move to page-utils
Search.getInitialProps = ({ platform, query }) => {
  const searchTermQueryVal =
    flattenHeaderOrParam(query[SEARCH_TERM_QUERY_PARAM]) ?? '';

  return {
    queryVariables: {
      noQuery: searchTermQueryVal === '',
      platform,
      userQuery: searchTermQueryVal,
    },
  };
};

Search.query = graphql`
  query Search_Query(
    $userQuery: String!
    $platform: String!
    $noQuery: Boolean!
  ) {
    currentUser {
      ...types_currentUser @relay(mask: false)
    }
    searchFor(userQuery: $userQuery, platform: $platform) @skip(if: $noQuery) {
      ...SearchResults_results
    }
  }
`;
