import { random } from 'faker';
import { createMountWrapperFactory } from 'tachyon-test-utils';
import { SEARCH_TERM_QUERY_PARAM } from '../../../config';
import { mockRouter } from '../../../routing/test-mocks';
import { InternalServerError } from '../../errors';
import { SearchContent } from './SearchContent';
import { Search } from '.';

jest.mock('../../../routing/useTachyonRouter', () => ({
  useTachyonRouter: () => mockRouter,
}));

jest.mock('./SearchContent', () => ({
  SearchContent: jest.fn(() => <div />),
}));

describe(Search, () => {
  const platform = random.alphaNumeric();
  const setup = createMountWrapperFactory(Search, () => ({
    currentUser: null,
    queryVariables: {
      noQuery: false,
      platform,
      userQuery: 'therealderekt',
    },
    searchFor: {
      ' $fragmentRefs': {
        SearchResults_results: true,
      },
    },
  }));

  it('renders internal error if results are invalid and there was a search term', () => {
    const { wrapper } = setup({
      queryVariables: { noQuery: false },
      searchFor: null,
    });

    expect(wrapper.find(InternalServerError)).toExist();
  });

  it('does not render internal error if the query was skipped because there was no search term', () => {
    const { wrapper } = setup({
      queryVariables: { noQuery: true },
      searchFor: null,
    });

    expect(wrapper.find(InternalServerError)).not.toExist();
  });

  it('renders if valid search term and results', () => {
    const { wrapper } = setup();
    expect(wrapper.find(SearchContent)).toExist();
  });

  describe(Search.getInitialProps!, () => {
    it('sets noQuery variable when the search query param is empty', () => {
      expect(
        Search.getInitialProps!({
          platform,
          query: { [SEARCH_TERM_QUERY_PARAM]: undefined },
        } as any),
      ).toEqual({
        queryVariables: {
          noQuery: true,
          platform,
          userQuery: '',
        },
      });

      expect(
        Search.getInitialProps!({
          platform,
          query: { [SEARCH_TERM_QUERY_PARAM]: '' },
        } as any),
      ).toEqual({
        queryVariables: {
          noQuery: true,
          platform,
          userQuery: '',
        },
      });
    });

    it('extracts the search term from query', () => {
      expect(
        Search.getInitialProps!({
          platform,
          query: { [SEARCH_TERM_QUERY_PARAM]: 'foo bar' },
        } as any),
      ).toEqual({
        queryVariables: {
          noQuery: false,
          platform,
          userQuery: 'foo bar',
        },
      });
    });
  });
});
