import { graphql } from 'react-relay/hooks';
import styled from 'styled-components';
import {
  usePublicPlatformName,
  useStaticEnvironment,
} from 'tachyon-environment';
import { useIntl } from 'tachyon-intl';
import { defaultPageviewTracking, getCurrentUser } from 'tachyon-page-utils';
import { useRequestInfo } from 'tachyon-relay';
import { SVGAsset } from 'twitch-core-ui';
import { DarkThemeMap } from 'twitch-core-ui-tokens';
import { OVERSCAN_PADDING_REM } from '../../../config';
import { RouteName, renderTachyonLink } from '../../../routing';
import type { SettingsMenuOption } from '../../common';
import { SettingsMenu, StarshotMain } from '../../common';
import { useStarshotVisualDebugMode } from '../../framework';
import type { StarshotPage } from '../types';
import type { Settings_QueryResponse } from './__generated__/Settings_Query.graphql';
import { useReloadOnClick } from './useReloadOnClick';

export type SettingsProps = Settings_QueryResponse;

const ScBuildId = styled.div`
  color: ${DarkThemeMap['color-text-button-disabled']};
  position: absolute;
  right: ${OVERSCAN_PADDING_REM}rem;
  top: 1.5rem;
  z-index: 1;
`;

export const KOREA_ISO_COUNTRY_CODE = 'KR';
const RELOAD_CLICKS_THRESHOLD = 5;
const RELOAD_CLICKS_TIMER_DURATION = 1000;

export const Settings: StarshotPage<{}, SettingsProps> = ({ currentUser }) => {
  const { formatMessage } = useIntl();
  const requestInfo = useRequestInfo();
  const {
    common: { appEnvironment },
  } = useStaticEnvironment();
  const { debugModeActive, toggleDebugMode } = useStarshotVisualDebugMode();
  const publicFacingPlatform = usePublicPlatformName();

  const isProd = appEnvironment === 'production';
  const isStaffUser = currentUser?.roles?.isStaff;
  const isKorea = requestInfo?.countryCode === KOREA_ISO_COUNTRY_CODE;

  const showDisableDebug = debugModeActive;
  const showEnableDebugMode = !debugModeActive && (!isProd || isStaffUser);
  const showDebugExperimentsManager = !isProd || debugModeActive;
  const showDebugPathExplorer = !isProd || debugModeActive;
  const showDebugEnvironmentSelector = !isProd || isStaffUser;
  // Always show the conditional user-facing options outside of production to
  // ensure we deal with effects on vertical height and interaction/ordering
  // with other options
  const showEntityInfo = !isProd || isKorea;

  const onBuildIdClick = useReloadOnClick(
    RELOAD_CLICKS_THRESHOLD,
    RELOAD_CLICKS_TIMER_DURATION,
  );

  const options: SettingsMenuOption[] = [
    {
      heading: formatMessage('Your Profile', 'SettingsPage'),
      icon: SVGAsset.NavProfile,
      interactive: {
        linkTo: '/deferToRenderLink' as const,
        renderLink: renderTachyonLink({
          route: RouteName.SettingsProfile,
        }),
      },
      subheading: formatMessage(
        'Visit your profile to sign in or out.',
        'SettingsPage',
      ),
    },
    {
      heading: formatMessage('Need Help?', 'SettingsPage'),
      icon: SVGAsset.FollowHollow,
      interactive: {
        linkTo: '/deferToRenderLink' as const,
        renderLink: renderTachyonLink({
          route: RouteName.SettingsHelp,
        }),
      },
      subheading: formatMessage(
        'Visit https://help.twitch.tv for help.',
        'SettingsPage',
      ),
    },
    {
      heading: formatMessage('Terms of Service', 'SettingsPage'),
      icon: SVGAsset.Edit,
      interactive: {
        linkTo: '/deferToRenderLink' as const,
        renderLink: renderTachyonLink({
          route: RouteName.SettingsTermsOfService,
        }),
      },
      subheading: formatMessage(
        'Visit https://www.twitch.tv/p/legal/terms-of-service to view the Terms of Service.',
        'SettingsPage',
      ),
    },
    {
      heading: formatMessage('Privacy Policy', 'SettingsPage'),
      icon: SVGAsset.Ghost,
      interactive: {
        linkTo: '/deferToRenderLink' as const,
        renderLink: renderTachyonLink({
          route: RouteName.SettingsPrivacy,
        }),
      },
      subheading: formatMessage(
        'Visit https://www.twitch.tv/p/legal/privacy-notice to view the Privacy Policy.',
        'SettingsPage',
      ),
    },
  ];

  if (showEntityInfo) {
    options.push({
      heading: formatMessage('Entity Information', 'SettingsPage'),
      icon: SVGAsset.Emoticons,
      interactive: {
        linkTo: '/deferToRenderLink' as const,
        renderLink: renderTachyonLink({
          route: RouteName.SettingsEntityInformation,
        }),
      },
    });
  }

  if (showDisableDebug) {
    options.push({
      heading: 'Disable Debug Mode',
      icon: SVGAsset.Keyboard,
      interactive: {
        onClick: toggleDebugMode,
      },
    });
  }

  if (showEnableDebugMode) {
    options.push({
      heading: 'Enable Debug Mode',
      icon: SVGAsset.Keyboard,
      interactive: {
        onClick: toggleDebugMode,
      },
    });
  }

  if (showDebugExperimentsManager) {
    options.push({
      heading: 'Experiments Manager (Debug)',
      icon: SVGAsset.Shovel,
      interactive: {
        linkTo: '/deferToRenderLink' as const,
        renderLink: renderTachyonLink({
          route: RouteName.SettingsDebugExperimentManager,
        }),
      },
    });
  }

  if (showDebugEnvironmentSelector) {
    options.push({
      heading: 'Environment Selector (Debug)',
      icon: SVGAsset.Rocket,
      interactive: {
        linkTo: '/deferToRenderLink' as const,
        renderLink: renderTachyonLink({
          route: RouteName.SettingsDebugEnvironmentSelector,
        }),
      },
    });
  }

  if (showDebugPathExplorer) {
    options.push({
      heading: 'App Path Navigator (Debug)',
      icon: SVGAsset.Externallink,
      interactive: {
        linkTo: '/deferToRenderLink' as const,
        renderLink: renderTachyonLink({
          route: RouteName.SettingsDebugPathNavigator,
        }),
      },
    });
  }

  return (
    <StarshotMain>
      <ScBuildId onClick={onBuildIdClick}>
        <p>
          {formatMessage(
            'Platform: {publicFacingPlatform}',
            { publicFacingPlatform },
            'SettingsPage',
          )}
        </p>
        <p>
          {formatMessage(
            'Build id: {buildId}',
            { buildId: process.env.BUILD_ID },
            'SettingsPage',
          )}
        </p>
      </ScBuildId>
      <SettingsMenu items={options} />
    </StarshotMain>
  );
};

Settings.currentUser = getCurrentUser;
Settings.displayName = 'Settings';
Settings.navigationBehavior = () => ({
  displayBackButton: true,
  displayNavMenu: true,
});
Settings.pageviewTracking = defaultPageviewTracking;

Settings.query = graphql`
  query Settings_Query {
    currentUser {
      ...types_currentUser @relay(mask: false)
    }
  }
`;
