import { datatype, random } from 'faker';
import type { AppEnvironment } from 'tachyon-environment';
import { createMockStaticEnvironmentContext } from 'tachyon-environment';
import type { RequestInfoContext } from 'tachyon-relay';
import { useRequestInfo } from 'tachyon-relay';
import { createMountWrapperFactory } from 'tachyon-test-utils';
import { SettingsMenu } from '../../common';
import {
  defaultStarshotVisualDebugModeRootContext,
  useStarshotVisualDebugMode,
} from '../../framework';
import { KOREA_ISO_COUNTRY_CODE, Settings } from '.';

const mockRequestInfo = (): RequestInfoContext['requestInfo'] => ({
  countryCode: 'AA',
  fromEU: false,
  ipAddress: null,
  isFromEEA: false,
});
jest.mock('tachyon-relay', () => ({
  ...jest.requireActual('tachyon-relay'),
  useRequestInfo: jest.fn(() => mockRequestInfo()),
}));
const mockUseRequestInfo = useRequestInfo as jest.Mock;

jest.mock('../../common', () => ({
  ...jest.requireActual('../../common'),
  SettingsMenu: () => null,
}));

jest.mock('../../framework', () => {
  const original = jest.requireActual('../../framework');
  return {
    ...original,
    useStarshotVisualDebugMode: jest.fn(
      () => original.defaultStarshotVisualDebugModeRootContext,
    ),
  };
});
const mockUseStarshotVisualDebugMode = useStarshotVisualDebugMode as jest.Mock;

describe(Settings, () => {
  function mockAppEnvironmentContext(
    appEnvironment: AppEnvironment,
  ): ReturnType<typeof createMockStaticEnvironmentContext> {
    const mock = createMockStaticEnvironmentContext();
    mock[1].common.appEnvironment = appEnvironment;
    return mock;
  }

  const setup = createMountWrapperFactory(
    Settings,
    () => ({
      currentUser: {
        id: datatype.uuid(),
        login: random.alphaNumeric(),
        roles: null,
      },
    }),
    { wrappingContexts: () => [mockAppEnvironmentContext('production')] },
  );

  describe('entity info option ', () => {
    const heading = 'Entity Information';

    it('does not render when not Korean country code and appEnvironment is production', () => {
      const { wrapper } = setup();
      expect(wrapper.find(SettingsMenu).prop('items')).not.toContainEqual(
        expect.objectContaining({ heading }),
      );
    });

    it('renders when Korean country code and appEnvironment is production', () => {
      mockUseRequestInfo.mockImplementationOnce(() => ({
        ...mockRequestInfo(),
        countryCode: KOREA_ISO_COUNTRY_CODE,
      }));

      const { wrapper } = setup();
      expect(wrapper.find(SettingsMenu).prop('items')).toContainEqual(
        expect.objectContaining({ heading }),
      );
    });

    it('renders when appEnvironment is staging', () => {
      const { wrapper } = setup(undefined, {
        wrappingContexts: [mockAppEnvironmentContext('staging')],
      });
      expect(wrapper.find(SettingsMenu).prop('items')).toContainEqual(
        expect.objectContaining({ heading }),
      );
    });
  });

  describe('disable debug mode', () => {
    const heading = 'Disable Debug Mode';

    it('does not render when debug mode is not active', () => {
      const { wrapper } = setup();
      expect(wrapper.find(SettingsMenu).prop('items')).not.toContainEqual(
        expect.objectContaining({ heading }),
      );
    });

    it('render when debug mode is active', () => {
      mockUseStarshotVisualDebugMode.mockImplementationOnce(() => ({
        ...defaultStarshotVisualDebugModeRootContext,
        debugModeActive: true,
      }));
      const { wrapper } = setup();
      expect(wrapper.find(SettingsMenu).prop('items')).toContainEqual(
        expect.objectContaining({ heading }),
      );
    });
  });

  describe('enable debug mode', () => {
    const heading = 'Enable Debug Mode';

    it('does not render when debug mode is active and appEnvironment is production', () => {
      mockUseStarshotVisualDebugMode.mockImplementationOnce(() => ({
        ...defaultStarshotVisualDebugModeRootContext,
        debugModeActive: true,
      }));
      const { wrapper } = setup();
      expect(wrapper.find(SettingsMenu).prop('items')).not.toContainEqual(
        expect.objectContaining({ heading }),
      );
    });

    it('does not render when debug mode is active and appEnvironment is staging', () => {
      mockUseStarshotVisualDebugMode.mockImplementationOnce(() => ({
        ...defaultStarshotVisualDebugModeRootContext,
        debugModeActive: true,
      }));

      const { wrapper } = setup(undefined, {
        wrappingContexts: [mockAppEnvironmentContext('staging')],
      });
      expect(wrapper.find(SettingsMenu).prop('items')).not.toContainEqual(
        expect.objectContaining({ heading }),
      );
    });

    it('renders when debug mode is not active and appEnvironment is staging', () => {
      const { wrapper } = setup(undefined, {
        wrappingContexts: [mockAppEnvironmentContext('staging')],
      });
      expect(wrapper.find(SettingsMenu).prop('items')).toContainEqual(
        expect.objectContaining({ heading }),
      );
    });

    it('renders when debug mode is not active and user is staff', () => {
      const { wrapper } = setup({
        currentUser: {
          roles: {
            isStaff: true,
          },
        },
      });
      expect(wrapper.find(SettingsMenu).prop('items')).toContainEqual(
        expect.objectContaining({ heading }),
      );
    });
  });

  describe('experiments manager', () => {
    const heading = 'Experiments Manager (Debug)';

    it('does not render when debug mode is not active and appEnvironment is production', () => {
      const { wrapper } = setup();
      expect(wrapper.find(SettingsMenu).prop('items')).not.toContainEqual(
        expect.objectContaining({ heading }),
      );
    });

    it('renders when debug mode is active and appEnvironment is production', () => {
      mockUseStarshotVisualDebugMode.mockImplementationOnce(() => ({
        ...defaultStarshotVisualDebugModeRootContext,
        debugModeActive: true,
      }));
      const { wrapper } = setup();
      expect(wrapper.find(SettingsMenu).prop('items')).toContainEqual(
        expect.objectContaining({ heading }),
      );
    });

    it('renders when debug mode not active and appEnvironment is staging', () => {
      const { wrapper } = setup(undefined, {
        wrappingContexts: [mockAppEnvironmentContext('staging')],
      });
      expect(wrapper.find(SettingsMenu).prop('items')).toContainEqual(
        expect.objectContaining({ heading }),
      );
    });
  });

  describe('environment selector', () => {
    const heading = 'Environment Selector (Debug)';
    it('does not render when user is not staff and appEnvironment is production', () => {
      const { wrapper } = setup();
      expect(wrapper.find(SettingsMenu).prop('items')).not.toContainEqual(
        expect.objectContaining({ heading }),
      );
    });

    it('renders the env selector option when current user is staff and appEnvironment is production', () => {
      const { wrapper } = setup({
        currentUser: {
          roles: {
            isStaff: true,
          },
        },
      });
      expect(wrapper.find(SettingsMenu).prop('items')).toContainEqual(
        expect.objectContaining({ heading }),
      );
    });

    it('renders when appEnvironment is staging', () => {
      const { wrapper } = setup(undefined, {
        wrappingContexts: [mockAppEnvironmentContext('staging')],
      });
      expect(wrapper.find(SettingsMenu).prop('items')).toContainEqual(
        expect.objectContaining({ heading }),
      );
    });
  });
});
