import { useEffect, useRef } from 'react';

// A "dev-mode" feature that enables a full page refresh if the build id
// is clicked threshold times within
// duration ms.
export function useReloadOnClick(
  threshold: number,
  duration: number,
): () => void {
  const reloadClicks = useRef(0);
  const reloadTimerHandle = useRef<number | null>(null);
  useEffect(() => () => {
    // ensure we don't leave this handle lying around
    if (reloadTimerHandle.current) {
      window.clearTimeout(reloadTimerHandle.current);
    }
  });

  function onClick(): void {
    // increment the click count
    reloadClicks.current++;

    // if we've tripped the threshold, clear the timeout and reload the page
    if (reloadClicks.current >= threshold) {
      if (reloadTimerHandle.current) {
        window.clearTimeout(reloadTimerHandle.current);
      }
      window.location.reload();
    }

    // if there isn't a current timer, start one that will reset everything once
    // the window closes
    if (!reloadTimerHandle.current) {
      reloadTimerHandle.current = window.setTimeout(() => {
        reloadClicks.current = 0;
        reloadTimerHandle.current = null;
      }, duration);
    }
  }

  return onClick;
}
