import { graphql } from 'react-relay/hooks';
import { useCurrentUser } from 'tachyon-auth';
import { useIntl } from 'tachyon-intl';
import type {
  ChannelInitialProps,
  ChannelPathParameters,
} from 'tachyon-page-utils';
import {
  channelIsFound,
  channelIsNotFoundServerside,
  channelPathGetInitialProps,
  defaultPageviewTracking,
  getCurrentUser,
} from 'tachyon-page-utils';
import { SVGAsset } from 'twitch-core-ui';
import {
  RouteName,
  createRedirectQuery,
  renderTachyonLink,
} from '../../../routing';
import type { SettingsMenuOption } from '../../common';
import { SettingsMenu, StarshotMain } from '../../common';
import { NotFoundError } from '../../errors';
import type { StarshotPage } from '../types';
import type { SettingsChannel_QueryResponse } from './__generated__/SettingsChannel_Query.graphql';

export type SettingsChannelProps = ChannelInitialProps &
  SettingsChannel_QueryResponse;

export const SettingsChannel: StarshotPage<
  ChannelInitialProps,
  SettingsChannelProps,
  ChannelPathParameters
> = ({ channel }) => {
  const { formatMessage } = useIntl();
  const { loggedIn } = useCurrentUser();

  if (!channelIsFound(channel)) {
    return <NotFoundError />;
  }

  const menuOptions: SettingsMenuOption[] = [
    {
      heading: formatMessage('Report Channel', 'SettingsChannel'),
      icon: SVGAsset.Edit,
      interactive: {
        linkTo: '/deferToRenderLink',
        renderLink: renderTachyonLink({
          route: RouteName.SettingsChannelReport,
          routeParams: {
            login: channel.login,
          },
        }),
      },
      subMenuIcon: true,
      subheading: formatMessage(
        'Is this channel violating ToS? Tell us about it!',
        'SettingsChannel',
      ),
    },
    {
      heading: formatMessage(
        'Not interested in this channel?',
        'SettingsChannel',
      ),
      icon: SVGAsset.NavCreative,
      interactive: {
        linkTo: '/deferToRenderLink',
        renderLink: loggedIn
          ? renderTachyonLink({
              route: RouteName.SettingsChannelNotInterested,
              routeParams: {
                login: channel.login,
              },
            })
          : renderTachyonLink({
              query: createRedirectQuery({
                route: RouteName.SettingsChannelNotInterested,
                routeParams: { login: channel.login },
              }),
              route: RouteName.Login,
            }),
      },
      subMenuIcon: true,
      subheading: formatMessage(
        "We won't recommend it to you anymore",
        'SettingsChannel',
      ),
    },
  ];

  return (
    <StarshotMain>
      <SettingsMenu items={menuOptions} />
    </StarshotMain>
  );
};

SettingsChannel.currentUser = getCurrentUser;
SettingsChannel.displayName = 'SettingsChannel';
SettingsChannel.getInitialProps = channelPathGetInitialProps;
SettingsChannel.isNotFoundServerside = channelIsNotFoundServerside;
SettingsChannel.navigationBehavior = () => ({
  displayBackButton: true,
});
SettingsChannel.pageviewTracking = defaultPageviewTracking;

SettingsChannel.query = graphql`
  query SettingsChannel_Query($login: String!) {
    currentUser {
      ...types_currentUser @relay(mask: false)
    }
    channel: user(login: $login) {
      id
      login
    }
  }
`;
