import { datatype, internet } from 'faker';
import { useCurrentUser } from 'tachyon-auth';
import { createShallowWrapperFactory } from 'tachyon-test-utils';
import {
  RouteName,
  createRedirectQuery,
  renderTachyonLink,
} from '../../../routing';
import { NotFoundError } from '../../errors';
import { SettingsChannel } from '.';

jest.mock('tachyon-auth', () => ({
  ...jest.requireActual('tachyon-auth'),
  useCurrentUser: jest.fn(() => ({ loggedIn: false })),
}));

jest.mock('../../../routing', () => ({
  ...jest.requireActual('../../../routing'),
  renderTachyonLink: jest.fn(() => () => <div />),
}));

const mockUseCurrentUser = useCurrentUser as jest.Mock;
const mockRenderTachyonLink = renderTachyonLink as jest.Mock;

describe(SettingsChannel, () => {
  const setup = createShallowWrapperFactory(SettingsChannel, () => ({
    channel: {
      id: datatype.uuid(),
      login: internet.userName(),
    },
    currentUser: null,
    queryVariables: {
      login: 'foO',
    },
  }));

  it('renders not found on missing channel', () => {
    const { wrapper } = setup({ channel: null });
    expect(wrapper.find(NotFoundError)).toExist();
  });

  describe('not interested button', () => {
    it('links to the SettingsChannelNotInterested page when logged in', () => {
      mockUseCurrentUser.mockImplementationOnce(() => ({
        loggedIn: true,
      }));
      const { props } = setup();

      expect(mockRenderTachyonLink).toHaveBeenCalledWith({
        route: RouteName.SettingsChannelNotInterested,
        routeParams: {
          login: props.channel!.login,
        },
      });
    });

    it('links to login page when not logged in', () => {
      mockUseCurrentUser.mockImplementationOnce(() => ({
        loggedIn: false,
      }));
      const { props } = setup();

      expect(mockRenderTachyonLink).toHaveBeenCalledWith({
        query: createRedirectQuery({
          route: RouteName.SettingsChannelNotInterested,
          routeParams: {
            login: props.channel!.login,
          },
        }),
        route: RouteName.Login,
      });
    });
  });
});
